/*= DIGISTANT 4462 ===========================================================*/
/*																				*/
/*  LabWindows Instrument Driver Version CVI 5.0.1                            	*/
/*  for Burster Digistant Typ 4462												*/
/*  Original Release: 	1.0 	Date: 2000.02.21                              	*/
/*  File Name:			bp4462.h												*/
/*  By: Thomas Meder                                                          	*/
/*  Originally written in C                                                   	*/
/*																				*/
/*  Modification History: 12.06.2002 by T. Meder 	Release 2.0           		*/
/*    The following functions were extended with parameter 'version' which is	*/
/*    a Flag for the new 60 Volts Digistant 4462 device Version					*/
/*		- bp4462_SetCalConfVolt													*/
/*		- bp4462_SetConfigRamp1													*/
/*		- bp4462_SetConfigListRamp2												*/
/*		- bp4462_SetConfigOutputStatus											*/
/* 		- bp4462_SetConfigOutputLimitV											*/
/*		- bp4462_SetConfigOutputLimitC											*/
/*		- bp4462_SetStoreIdOutputValue											*/
/*		- bp4462_SetConfigRangeMode												*/
/*		- bp4462_SetConfigDeltaValue											*/
/*																				*/
/*  Modification History: 14.11.2003 by T. Meder 	Release 2.1           		*/
/*    The temperature output constant values of K, C and F were changed.		*/
/*											*/
/*  Modification History: 24.08.2004 by T. Meder 	Release 2.2           		*/
/*    Changed Variable names because of consolidated compiler restrictions.		*/
/*==============================================================================*/

/*---- LabWindows DOS Header----*/
#ifdef LW_DOS             /* Define fuer DOS-Umgebung in demo4462.mk*/
 #include "\lw\include\lwsystem.h"
 #include "\lw\include\gpib.h"
 #include "\lw\include\rs232.h"
 #include "\lw\include\formatio.h"
 #include <string.h>
#else
 /*---- LabWindows CVI Header----*/
 #include <utility.h>
 #include <rs232.h>
 #include <formatio.h>
 #include <gpib.h>
 #include <ansi_c.h>
#endif

#include "bp4462.h"

/*=========================== STATIC VARIABLES ===========================================*/
static int instr_cnt;
static int interface;    			  	  /* 0 = GIPB; 1 = Serial */
static int bd[bp4462_MAX_INSTR + 1];	  /* Gpib Board */
static int address[bp4462_MAX_INSTR + 1]; /* Gipb-address */
static int port_no = 1;					  /* default: Port 1 */
static int setup_first ;  				  /* remark the first initial */ 
int bp4462_err;					  		  /* Device error */ 

static char string_value[40], strv1[20], stru1[20],   /* For Function strsplit */ 
		    strv2[20], stru2[2], strv3[20], stru3[20];  

static char cmd[bp4462_MAX_CMD+1];
static char buf[bp4462_MAX_CMD+1];
static char out_buf[bp4462_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp4462_MAX_CMD+1];        /* buffer for serial receive */
static int  dev_port[bp4462_MAX_INSTR+1];	 /* Device Port */
static int  int_level[4];					 /* RS 232 Interrupt Level Array */

static char* ramp1_wave[2];
static char* ramp2_wave[2]; 
static char* ramp_start[2];
static char* unit_type[8];
static char* tc_unit[6];
static char* step_mode[2]; 
static char* man_range[4];
static char* range_str[4];
static char* sign_str[2];
static char* output_str[3];
static char* rj_type[2];
static char* state_str[2];
static char* unit_str[2];
static char* tc[14];
static char* tc_scale[2];
static char* output_mode[3]; 
static char* sense_mode[2];
static char* display_mode[6]; 
static char* set_delta[2];
static char* delta_value[3];


/*================================ UTILITY ROUTINES ==================================*/

int _VI_FUNC bp4462_open_instr_serial (int, long, int, int, int); 
int _VI_FUNC bp4462_close_instr_serial (int);
int _VI_FUNC bp4462_close_instr_gpib (int);
int _VI_FUNC bp4462_open_instr_gpib (int); 
int _VI_FUNC bp4462_device_closed_gpib (int);
int _VI_FUNC bp4462_write_msg_gpib (int, char*, int);
int _VI_FUNC bp4462_read_msg_gpib (int, char*, int);
void _VI_FUNC bp4462_strsplit (char[], char*, char*, char*, char*, char*, char*); 
void _VI_FUNC bp4462_valuesplit (char[], double*, double*, double*, double*, double*);
int _VI_FUNC bp4462_invalid_integer_range (int, int, int, int);
int _VI_FUNC bp4462_invalid_real_range (double, double, double, int);
int _VI_FUNC bp4462_invalid_integer_range (int, int, int, int);
int _VI_FUNC bp4462_send_rs232 (int, char* );
int _VI_FUNC bp4462_receive_rs232 (int, char*, char* );
int _VI_FUNC bp4462_receive_rs232_message (int, char*);
void bp4462_setup_arrays (void); 

/*=========================================================================*/
/* This function opens a RS-com port for the instrument module             */
/*=========================================================================*/

int _VI_FUNC bp4462_InitSerial(int port, int baud_rate,int data_config,
                   			int reset_flag, int *instr_ID)
{
	int ID;
	int databits;
	int stopbits;
	int parity;
	
	interface = bp4462_SERIAL;	/* Serial */

    bp4462_err = 0;
    bp4462_setup_arrays ();

    if (bp4462_invalid_integer_range (port, 1,4, -1) != 0)				   
     	return bp4462_err;
    if (bp4462_invalid_integer_range (baud_rate, 300, 38400, -2) != 0)
     	return bp4462_err; 
    if (bp4462_invalid_real_range (data_config, 0, 7, -3) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (reset_flag, 0,1, -4) != 0)
     	return bp4462_err;
     	
    port_no = port;
   
    if ((data_config >= 0) && (data_config <= 3)) 		/* 8 Databits */
     	databits = 8;
    else
     	databits = 7;
     
    if ((data_config == 3) || (data_config == 6)) 		/* 2 Stopbits */
     	stopbits = 2;
    else
     	stopbits = 1;
    
    if ((data_config == 1) || (data_config == 4))      	/* Even Parity */
     	parity = 2;
    else if ((data_config == 2) || (data_config == 5)) 	/* Odd Parity  */
     	parity = 1;
    else
     	parity = 0;	    								/* No Parity   */
    
	ID = bp4462_open_instr_serial (port, baud_rate, parity, databits, stopbits);
	 
	if (ID <= 0)
	  	return bp4462_err;
	  	
	/* Reset Device? */  
	if (reset_flag == 1)
	  	{
	  	Delay (1);
	  	if (bp4462_send_scpi(ID, "*RST")!=0)	   
	    	{
	    	bp4462_close_instr_serial(ID);
	    	return bp4462_err;
	    	}
	  	}	   
	
	*instr_ID = ID;
	
    return (0);
} 
/*=========================================================================*/
/* This function opens the instrument, queries the instrument for its ID   */
/* and initializes the instrument to a known state.                        */
/*=========================================================================*/
int _VI_FUNC bp4462_InitGpib (int addr, int reset, int *instr_ID)
{
    int ID;
    bp4462_err = 0;
    bp4462_setup_arrays ();
    
    interface = bp4462_GPIB;   /* GPIB */

    if (bp4462_invalid_integer_range (addr, 0, 31,  -1) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (reset, 0, 1,  -2) != 0)
      	return bp4462_err;

	ID = bp4462_open_instr_gpib(addr);
	if (ID <= 0)
	  	return bp4462_err;
	  
/*	ibtmo (bd[ID], time_out);  */

    /*  terminate read on LF  */
    if (ibeos (bd[ID], 0x40a) < 0)
      	{
      	bp4462_err = 237;
      	return bp4462_err;
      	}
    
	ibconfig (bd[ID],IbcREADDR ,1);   /* Repeat Addressing */
	
	 if (reset == 1)
	 	{
	 	Delay (1);
	 	if (bp4462_send_scpi (ID, "*RST") != 0)
        	{
	        bp4462_close_instr_gpib (ID);
	        return bp4462_err;
        	}
        }	
      
    *instr_ID = ID;
 
    return(0);
}
/* ========================================================================= */
/*  This function sets Ramp1    033 034 035 036 037 038 039 040 041 042 043  */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 		*/
/*  		in Release V 2.0   21.05.2002 by T. Meder						*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigRamp1 (int instr_ID, int waveform, int number_of_counts, 
                      int step_time_h, int step_time_min, double step_time_sec, 
                      int unit, double step_value, double start_value, double stop_value, int version)
{
  	char unit_char[10];
  	int minVolt, maxVolt, minCurrUA, maxCurrUA, minCurrMA, maxCurrMA;
  	
  	if (version == 1)			// 60 Volt device version
  	{
  		minVolt = bp4462_MIN60_V;
  		maxVolt = bp4462_MAX60_V;
  		minCurrUA = bp4462_MIN60_UA;
  		maxCurrUA = bp4462_MAX60_UA;
  		minCurrMA = bp4462_MIN60_MA;
  		maxCurrMA = bp4462_MAX60_MA;
  	}
  	else						// 30 Volt device version 
  	{
  		minVolt = bp4462_MIN30_V;
    	maxVolt = bp4462_MAX30_V;
  		minCurrUA = bp4462_MIN30_UA;
  		maxCurrUA = bp4462_MAX30_UA;
  		minCurrMA = bp4462_MIN30_MA;
  		maxCurrMA = bp4462_MAX30_MA;
    }
   
  	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (waveform, 0, 1,  -2) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (unit, 0, 7,  -3) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (number_of_counts, 0, 99,  -4) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_real_range (step_value, 0.2, 99999.9,  -5) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (step_time_h, 0, 99,  -6) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (step_time_min, 0, 59,  -7) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_real_range (step_time_sec, 0, 59.9,  -8) != 0)
    	return bp4462_err;
     
  	if (bp4462_send_scpi (instr_ID, ramp1_wave[waveform]) != 0)
    	return bp4462_err;
       
  	Fmt (cmd, "%s<SOUR:SWE:COUN %i",number_of_counts);
  	if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;

  	Fmt (cmd, "%s<SOUR:SWE:DWEL %d,%d,%f",step_time_h, step_time_min, step_time_sec);
  	if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;


  	if ((unit >= 0) && (unit <= 2)) 	/* Voltage*/
		{
		if (unit == 0) 					/* V */
	  		{
	  		Fmt (unit_char,"%s<UV");
      		if (bp4462_invalid_real_range (start_value, bp4462_MIN_UV,bp4462_MAX_UV,  -9) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (stop_value, bp4462_MIN_UV,bp4462_MAX_UV,  -10) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (step_value, bp4462_MIN_UV,bp4462_MAX_UV,  -11) != 0)
        		return bp4462_err;
      		}
    	else						
		if (unit == 1) 					/* mV */
	  		{
	  		Fmt (unit_char,"%s<MV");
      		if (bp4462_invalid_real_range (start_value, bp4462_MIN_MV,bp4462_MAX_MV,  -9) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (stop_value, bp4462_MIN_MV,bp4462_MAX_MV,  -10) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (step_value, bp4462_MIN_MV,bp4462_MAX_MV,  -11) != 0)
        		return bp4462_err;
      		}
      	else							/* Volt */
      		{
      		Fmt(unit_char,"%s<V");
      		if (bp4462_invalid_real_range (start_value, minVolt,maxVolt,  -9) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (stop_value, minVolt,maxVolt,  -10) != 0)
        		return bp4462_err;
      		if (bp4462_invalid_real_range (step_value, minVolt,maxVolt,  -11) != 0)
        		return bp4462_err;
      		}  
      		
    	Fmt (cmd, "%s<SOUR:VOLT:STAR %f%s",start_value,unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	Fmt (cmd, "%s<SOUR:VOLT:STOP %f%s",stop_value,unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	Fmt (cmd, "%s<SOUR:VOLT:STEP %f%s",step_value,unit_char);
    		if (bp4462_send_scpi (instr_ID, cmd) != 0)
      	return bp4462_err;
    	}
    	
  	else if ((unit == 3)||(unit == 4))  /* Current*/
    	{
    	if (unit == 3)		  			/* A */
    		{
    		Fmt(unit_char,"%s<UA");
	    	if (bp4462_invalid_real_range (start_value, minCurrUA,maxCurrUA,  -9) != 0)
	      		return bp4462_err;
	    	if (bp4462_invalid_real_range (stop_value, minCurrUA,maxCurrUA,  -10) != 0)
	      		return bp4462_err;
	    	if (bp4462_invalid_real_range (step_value, minCurrUA,maxCurrUA,  -11) != 0)
	      		return bp4462_err;
	      	}	
    	else		  					/* mA */
    		{
    		Fmt(unit_char,"%s<MA");
	    	if (bp4462_invalid_real_range (start_value, minCurrMA,maxCurrMA,  -9) != 0)
	      		return bp4462_err;
	    	if (bp4462_invalid_real_range (stop_value, minCurrMA,maxCurrMA,  -10) != 0)
	      		return bp4462_err;
	    	if (bp4462_invalid_real_range (step_value, minCurrMA,maxCurrMA,  -11) != 0)
	      		return bp4462_err;
	      	}	
     
    	Fmt (cmd, "%s<SOUR:CURR:STAR %f%s",start_value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
     		return bp4462_err;
    	Fmt (cmd, "%s<SOUR:CURR:STOP %f%s",stop_value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	Fmt (cmd, "%s<SOUR:CURR:STEP %f%s",step_value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	}
  	else if ((unit >= 5) && (unit <= 7)) /* TC */
    	{
		switch(unit)
	  		{
	  		case 5:						/* C */
        		if (bp4462_invalid_real_range (start_value, bp4462_MIN_GRD_C,bp4462_MAX_GRD_C,  -9) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (stop_value, bp4462_MIN_GRD_C,bp4462_MAX_GRD_C,  -10) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (step_value, bp4462_MIN_GRD_C,bp4462_MAX_GRD_C,  -11) != 0)
          			return bp4462_err;
	    		break;
	  		case 6:						/* F */
        		if (bp4462_invalid_real_range (start_value, bp4462_MIN_GRD_F,bp4462_MAX_GRD_F,  -9) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (stop_value, bp4462_MIN_GRD_F,bp4462_MAX_GRD_F,  -10) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (step_value, bp4462_MIN_GRD_F,bp4462_MAX_GRD_F,  -11) != 0)
          			return bp4462_err;
	    		break;
	  		case 7:						/* K */
        		if (bp4462_invalid_real_range (start_value, bp4462_MIN_K,bp4462_MAX_K,  -9) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (stop_value, bp4462_MIN_K,bp4462_MAX_K,  -10) != 0)
          			return bp4462_err;
        		if (bp4462_invalid_real_range (step_value, bp4462_MIN_K,bp4462_MAX_K,  -11) != 0)
          			return bp4462_err;
	    		break;
	  		}  
    
    	Fmt (cmd, "%s<SOUR:TCO:STAR %f%c",start_value, unit_type[unit]);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;

    	Fmt (cmd, "%s<SOUR:TCO:STOP %f%c",stop_value, unit_type[unit]);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0) 
      		return bp4462_err;

    	Fmt (cmd, "%s<SOUR:TCO:STEP %f%c",step_value, unit_type[unit]);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    }
  	return 0;
}


/* ================================================================================================ */
/*  This function gets the configuration of the Ramp1  033 034 035 036 037 038 039 040 041 042 043  */
/* ================================================================================================ */
int _VI_FUNC bp4462_GetConfigRamp1 (int instr_ID, int* waveform, int* number_of_counts, 
					int* step_time_h, int* step_time_min, double* step_time_sec,
					int* step_unit, double* step_value, int* start_unit, 
					double* start_value, int* stop_unit, double* stop_value)
{
	char string;
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
     
  	Fmt	(cmd,"%s<sour:swe:wav?");
  	if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
   
  	if (FindPattern (buf, 0, -1, "SAWT", 0, 0) >= 0)	/* waveform? */
		*waveform = 0;
  	else if(FindPattern (buf, 0, -1, "TRI", 0, 0) >= 0)
		*waveform = 1;
  	else return (-2);	
	
  	Fmt (cmd,"%s<sour:swe:coun?");					   	/* counts of execution */
  	if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
  	Scan (buf,"%s>%d",number_of_counts);   
	
  	Fmt (cmd,"%s<sour:swe:dwel?");					   	/* Valuestring time format: hh,mm,ss.s */
  	if (bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
    	return bp4462_err;
	
  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3); 
 
  	Scan(strv1, "%s>%i", step_time_h); 		   
  	Scan(strv2, "%s>%i", step_time_min);
  	Scan(strv3, "%s>%f", step_time_sec);

  	Fmt(cmd,"%s<sour:volt:start?");				       	/* start value and unit of Ramp1 */
  	if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
    
  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3); 

  	Scan(strv1, "%s>%f", start_value); 					/* Find the unit */

	if (FindPattern (stru1, 0, -1, "UV", 0, 0) != -1)
		*start_unit = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) != -1)
		*start_unit = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) != -1)
		*start_unit = 2;
	else if (FindPattern (stru1, 0, -1, "UA", 0, 0) != -1)
		*start_unit = 3;
	else if (FindPattern (stru1, 0, -1, "MA", 0, 0) != -1)
		*start_unit = 4;
	else if (FindPattern (stru1, 0, -1, "C", 0, 0) != -1)
		*start_unit = 5;
	else if (FindPattern (stru1, 0, -1, "F", 0, 0) != -1)
		*start_unit = 6;
	else if (FindPattern (stru1, 0, -1, "K", 0, 0) != -1)
		*start_unit = 7;
	else return (-3);	

  	Fmt(cmd,"%s<sour:volt:stop?");				 		/* get the stop value and unit of Ramp1 */
  	if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
    	return bp4462_err;
     
  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3); 

  	Scan(strv1, "%s>%f", stop_value); 		   
  	
	if (FindPattern (stru1, 0, -1, "UV", 0, 0) != -1)
		*stop_unit = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) != -1)
		*stop_unit = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) != -1)
		*stop_unit = 2;
	else if (FindPattern (stru1, 0, -1, "UA", 0, 0) != -1)
		*stop_unit = 3;
	else if (FindPattern (stru1, 0, -1, "MA", 0, 0) != -1)
		*stop_unit = 4;
	else if (FindPattern (stru1, 0, -1, "C", 0, 0) != -1)
		*stop_unit = 5;
	else if (FindPattern (stru1, 0, -1, "F", 0, 0) != -1)
		*stop_unit = 6;
	else if (FindPattern (stru1, 0, -1, "K", 0, 0) != -1)
		*stop_unit = 7;
	else return (-4);	
  				   
  	Fmt(cmd,"%s<sour:volt:step?");						/* get the step value and unit of Ramp1 */
  	if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
    return bp4462_err;
     
  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3); 

  	Scan(strv1, "%s>%f", step_value); 		   
  	

	if (FindPattern (stru1, 0, -1, "UV", 0, 0) != -1)
		*step_unit = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) != -1)
		*step_unit = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) != -1)
		*step_unit = 2;
	else if (FindPattern (stru1, 0, -1, "UA", 0, 0) != -1)
		*step_unit = 3;
	else if (FindPattern (stru1, 0, -1, "MA", 0, 0) != -1)
		*step_unit = 4;
	else if (FindPattern (stru1, 0, -1, "C", 0, 0) != -1)
		*step_unit = 5;
	else if (FindPattern (stru1, 0, -1, "F", 0, 0) != -1)
		*step_unit = 6;
	else if (FindPattern (stru1, 0, -1, "K", 0, 0) != -1)
		*step_unit = 7;
	else return (-5);	
  	
  	return(0);
}

/* ========================================================================= */
/*  This function  sets Ramp2    045 046                                     */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigRamp2(int instr_ID, int waveform,int count,int points, int list)
{
  	bp4462_err = 0;
			    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (waveform, 0, 1, -2) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (count, 0, 99, -3) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (points, 2, 31, -4) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (list, 1, 31, -5) != 0)
     	return bp4462_err;

    if (bp4462_send_scpi(instr_ID, ramp2_wave[waveform]) != 0)
      	return bp4462_err;

    Fmt (cmd, "%s<SOUR:LIST:COUN %i",count);
    if (bp4462_send_scpi(instr_ID, cmd) != 0)
      	return bp4462_err;

    Fmt (cmd, "%s<SOUR:LIST:POIN %i",points);
    if (bp4462_send_scpi(instr_ID, cmd) != 0)
      	return bp4462_err;

    Fmt (cmd, "%s<SOUR:LIST:NUMB %i",list);
    if (bp4462_send_scpi(instr_ID, cmd) != 0)
      	return bp4462_err;
    
    return 0;
}

/*=============================================================================*/
/* This function gets the configuration of Ramp2   045 046                     */
/*=============================================================================*/
int _VI_FUNC bp4462_GetConfigRamp2(int instr_ID, int list, int *waveform, int *count, int* points)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd,"%s<SOUR:LIST:WAV?");
    if (bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
       
	if (FindPattern (buf, 0, -1, "SAWT", 0, 0) >= 0)
		*waveform = 0;
	else if(FindPattern (buf, 0, -1, "TRI", 0, 0) >= 0)
		*waveform = 1;
	else return (-2);
		
    Fmt (cmd,"%s<SOUR:LIST:COUN?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
    Scan (buf, "%s>%d", count);
    
    Fmt (cmd,"%s<SOUR:LIST:POIN?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
    Scan (buf, "%s>%d", points); 
    
    Fmt (cmd,"%s<SOUR:LIST:NUMB?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
    Scan (buf, "%s>%d", list); 
  
	return (0);
}
 
/* ========================================================================= */
/*  This function  sets Ramp2 list   046 047 049 050 051                     */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigListRamp2 (int instr_ID, int unit, int list_no, double value,
									   int step_time_h, int step_time_min, double step_time_sec, int version)
									   
{
    char unit_char[10];
  	int minVolt, maxVolt, minCurrUA, maxCurrUA, minCurrMA, maxCurrMA;
  	
  	if (version == 1)			// 60 Volt device version
  	{
  		minVolt = bp4462_MIN60_V;
  		maxVolt = bp4462_MAX60_V;
  		minCurrUA = bp4462_MIN60_UA;
  		maxCurrUA = bp4462_MAX60_UA;
  		minCurrMA = bp4462_MIN60_MA;
  		maxCurrMA = bp4462_MAX60_MA;
  	}
  	else						// 30 Volt device version 
  	{
  		minVolt = bp4462_MIN30_V;
    	maxVolt = bp4462_MAX30_V;
  		minCurrUA = bp4462_MIN30_UA;
  		maxCurrUA = bp4462_MAX30_UA;
  		minCurrMA = bp4462_MIN30_MA;
  		maxCurrMA = bp4462_MAX30_MA;
    }
    
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (unit, 0, 7,  -2) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (list_no, 1, 30,  -3) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (step_time_h, 0, 99,  -4) != 0)
     	return bp4462_err;
    if (bp4462_invalid_integer_range (step_time_min, 0, 59,  -5) != 0)
     	return bp4462_err;
    if (bp4462_invalid_real_range (step_time_sec, 0, 59.9,  -6) != 0)
     	return bp4462_err;

    Fmt (cmd, "%s<SOUR:LIST:NUMB %d",list_no);
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
     	return bp4462_err;

    Fmt (cmd, "%s<SOUR:LIST:DWEL %d,%d,%f",step_time_h, step_time_min, step_time_sec);
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
     	return bp4462_err;

  	if ((unit >= 0) && (unit <= 2)) 	/* Voltage*/
		{
		if (unit == 0) 					/* V */
	  		{
	   		Fmt (unit_char,"%s<UV");   
       		if (bp4462_invalid_real_range (value, bp4462_MIN_UV,bp4462_MAX_UV,  -7) != 0)
        		return bp4462_err;
	  		} 
		if (unit == 1) 					/* mV */
	  		{
	   		Fmt (unit_char,"%s<MV");   	
       		if (bp4462_invalid_real_range (value, bp4462_MIN_MV,bp4462_MAX_MV,  -7) != 0)
        		return bp4462_err;
	  		} 
	  	else
	  		{
	   		Fmt(unit_char,"%s<V");  	/* V */
       		if (bp4462_invalid_real_range (value, minVolt,maxVolt,  -7) != 0)
        		return bp4462_err;
	  		} 

      	Fmt (cmd, "%s<SOUR:LIST:VOLT %f%s", value, unit_char);
      	if (bp4462_send_scpi (instr_ID, cmd) != 0)
       		return bp4462_err;
    	}
    else if ((unit == 3)||(unit == 4))  /* Current*/
    	{
    	if (unit == 3)			   		/* A */
    		{
    		Fmt(unit_char,"%s<UA");
	      	if (bp4462_invalid_real_range (value, minCurrUA,maxCurrUA,  -7) != 0)
	       		return bp4462_err;
	       	}
	    else   							/* mA */
    		{
    		Fmt(unit_char,"%s<MA");
	      	if (bp4462_invalid_real_range (value, minCurrMA,maxCurrMA,  -7) != 0)
	       		return bp4462_err;
	       	}
	    

      	Fmt (cmd, "%s<SOUR:LIST:CURR %fMA",value, unit_char);  	/* mA */
      	if (bp4462_send_scpi (instr_ID, cmd) != 0)
       		return bp4462_err;
    	}
    else if ((unit >= 5) && (unit <= 7)) 						/* TC */
    	{
	 	switch (unit)
	 		{
	  		case 3:					/* C */
       			if (bp4462_invalid_real_range (value, bp4462_MIN_GRD_C,bp4462_MAX_GRD_C,  -7) != 0)
        			return bp4462_err;
	   			break;
	  		case 4:					/* F */
       			if (bp4462_invalid_real_range (value, bp4462_MIN_GRD_F,bp4462_MAX_GRD_F,  -7) != 0)
        			return bp4462_err;
	   			break;
	  		case 5:					/* K */
       			if (bp4462_invalid_real_range (value, bp4462_MIN_K,bp4462_MAX_K,  -7) != 0)
        		return bp4462_err;
	   			break;
	 		}  
    
     	Fmt (cmd, "%s<SOUR:LIST:TCO %f%s", value, unit_type[unit]);
     	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    }
    return 0;
}

/* ========================================================================= */
/*  This function gets the configuration of the Ramp2 list 046 047 048       */
/* ========================================================================= */
int _VI_FUNC bp4462_GetConfigListRamp2(int instr_ID, int list_no, int* unit, double* value,
					int* step_time_h, int* step_time_min, double* step_time_sec)
{   char string_value;
	
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd, "%s<SOUR:LIST:NUMB %d", list_no);
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
     	return bp4462_err;
	 
    Fmt (cmd,"%s<sour:list:dwel?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;

  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3) ;

  	Scan (strv1, "%s>%i", step_time_h); 
  	Scan (strv2, "%s>%i", step_time_min); 
  	Scan (strv3, "%s>%f", step_time_sec); 
	
    Fmt (cmd,"%s<SOUR:LIST:VAL?");				 /* Get the delta value and unit of list Ramp2 */
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     return bp4462_err;
        
	if (FindPattern (buf, 0, -1, "UV", 0, 0) != -1)
		*unit = 0;
	else if (FindPattern (buf, 0, -1, "MV", 0, 0) != -1)
		*unit = 1;
	else if (FindPattern (buf, 0, -1, "V", 0, 0) != -1)
		*unit = 2;
	else if (FindPattern (buf, 0, -1, "UA", 0, 0) != -1)
		*unit = 3;
	else if (FindPattern (buf, 0, -1, "MA", 0, 0) != -1)
		*unit = 4;
	else if (FindPattern (buf, 0, -1, "C", 0, 0) != -1)
		*unit = 5;
	else if (FindPattern (buf, 0, -1, "F", 0, 0) != -1)
		*unit = 6;
	else if (FindPattern (buf, 0, -1, "K", 0, 0) != -1)
		*unit = 7;
	else return (-2);	
		
	return(0);
}

/* ========================================================================= */
/*  This function set the ramp start   056                                   */
/* ========================================================================= */
int _VI_FUNC bp4462_SetRampStart (int instr_ID, int start_mode)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (start_mode, 0, 1,  -2) != 0)
     	return bp4462_err;

    if (bp4462_send_scpi (instr_ID, ramp_start[start_mode]) != 0)
     	return bp4462_err;

    return 0;
}
/* ========================================================================= */
/*  This function set one direction step     057                             */
/* ========================================================================= */
int _VI_FUNC bp4462_SetRampStep (int instr_ID, int direction_step)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (direction_step, 0, 1,  -2) != 0)
     	return bp4462_err;

    if (bp4462_send_scpi(instr_ID, step_mode[direction_step]) != 0)
     	return bp4462_err;

    return 0;
}

/*=========================================================================*/
/* This function sets the configuration of auto/man-range          020     */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/*=========================================================================*/
int _VI_FUNC bp4462_SetConfigRangeMode (int instr_ID, int range_mode, int version)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_invalid_integer_range (range_mode, 0, 3,  -2) != 0)
    	return bp4462_err;

	if (range_mode == 0)	/* Auto Range */		
		Fmt (cmd, "%s<SOUR:VOLT:RANG:AUTO 1");
		
	if (range_mode >= 1)	/* Manual Range */		
		Fmt (cmd, "%s<SOUR:VOLT:RANG:AUTO 0");
		
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;
      	
    if (range_mode >= 1)	/* Manual Range --> send Range Value */
    	if ((version == 1)&&(range_mode == 3))
    	{
    		if (bp4462_send_scpi (instr_ID, man_range[range_mode]) != 0)
    			return bp4462_err;
    	}
    	else
    	{
    		if (bp4462_send_scpi (instr_ID, man_range[range_mode -1]) != 0)
    			return bp4462_err;
    	}
    return (0);
}

/*=========================================================================*/
/* This function gets the configuration of auto/man-range         020      */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigRangeMode (int instr_ID, int *range_mode)
{
	int value;
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd,"%s<SOUR:VOLT:RANG:AUTO?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
    
    Scan (buf, "%s>%d", &value); 
    
    if (value == 0)		/* Manual Range */
    	{
    	/* Get Manual Range Value */
	  	Fmt (cmd,"%s<SOUR:VOLT:RANG?");				 
	  	if (bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
	    	return bp4462_err;
	
	  	if (FindPattern (buf, 0, -1, "300MV", 0, 0) >= 0)
	 		*range_mode = 1;
	  	else if (FindPattern (buf, 0, -1, "3V", 0, 0) >= 0)
	 		*range_mode = 2;
	  	else if (FindPattern (buf, 0, -1, "30V", 0, 0) >= 0)
	 		*range_mode = 3;
	  	else return (-2); 
	  	}
	else *range_mode = 0; 		/* Auto Range */
    
	return (0);
}

/* ========================================================================= */
/*  This function sets the RJ-Man-Temperature and unit       054             */
/* ========================================================================= */
int _VI_FUNC bp4462_SetRjTempValueMan (int instr_ID, int unit, double value)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    if (bp4462_invalid_integer_range (unit, 5, 7, -2) != 0)
     	return bp4462_err;

    switch (unit)
		{
		case 5:		/* C */
    		if (bp4462_invalid_real_range (value, bp4462_MIN_RJ_GRD_C, bp4462_MAX_RJ_GRD_C,  -3) != 0)
       			return bp4462_err;
			break;
		case 6:		/* F */
    		if (bp4462_invalid_real_range (value, bp4462_MIN_RJ_GRD_F, bp4462_MAX_RJ_GRD_F,  -3) != 0)
       			return bp4462_err;
			break;
		case 7:		/*  K */
    		if (bp4462_invalid_real_range (value, bp4462_MIN_RJ_K, bp4462_MAX_RJ_K,  -3) != 0)
      			return bp4462_err;
			break;
		}  
    
    Fmt (cmd, "%s<SENS:TCO:REFJ:TMAN %f%s",value, unit_type[unit]);
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;
    
    return (0);
}

/*=========================================================================*/
/* This function gets the  Rj-Manual-Temperature value and unit    054     */
/*=========================================================================*/
int _VI_FUNC bp4462_GetRjTempValueMan (int instr_ID, int *unit, double* value)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<sens:tco:refj:tman?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
		
	if (FindPattern (buf, 0, -1, "C", 0, 0) >= 0)
		*unit = 5;
	else if (FindPattern (buf, 0, -1, "F", 0, 0) >= 0)
		*unit = 6;
	else if (FindPattern (buf, 0, -1, "K", 0, 0) >= 0)
		*unit = 7;
		
  	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3); 
 
	Scan (strv1, "%s>%f", value); 		   

	return (0);
}

/*=========================================================================*/
/* This function sets the configuration of the RJ type  053                */
/*=========================================================================*/
int _VI_FUNC bp4462_SetConfigRjType (int instr_ID, int type)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (type, 0, 1,  -2) != 0)
     	return bp4462_err;

    if (bp4462_send_scpi (instr_ID, rj_type[type]) != 0)
      	return bp4462_err;

	return (0);
}

/*=========================================================================*/
/* This function gets the configuration of the Rj-type  053                */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigRjType(int instr_ID, int *type)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd,"%s<sens:tco:refj?");
    if (bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	
	if (FindPattern (buf, 0, -1, "RJ-MAN", 0, 0) >= 0)
		*type = 0;
	else if (FindPattern (buf, 0, -1, "RJ-EXT", 0, 0) >= 0)
		*type = 1;
	else return (-2);	

	return (0);
}

/*=========================================================================*/
/* This function gets the External RJ-Temperature and unit   055           */
/*=========================================================================*/
int _VI_FUNC bp4462_GetRjTempValue (int instr_ID, int *unit, double* value)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd,"%s<sens:tco:refj:temp?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;

	if (FindPattern (buf, 0, -1, "C", 0, 0) >= 0)
		*unit = 5;
	else if(FindPattern (buf, 0, -1, "F", 0, 0) >= 0)
		*unit = 6;
	else if(FindPattern (buf, 0, -1, "K", 0, 0) >= 0)
		*unit = 7;
	else return (-2);	
		
  	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3);
  	
  	Scan (strv1, "%s>%f", value); 
	
	return (0);
}

/*=========================================================================*/
/* This function gets the extern TC-voltage  066                           */
/*=========================================================================*/
int _VI_FUNC bp4462_GetTcVoltageExtern (int instr_ID, double* voltage_ut, int unit_ut, 
										double* voltage_u0, int unit_u0)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<CALC:TCO:UT?");
    if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
  	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3);
  	
	Scan (strv1,"%s>%f", voltage_ut);
	
	if (FindPattern (stru1, 0, -1, "UV", 0, 0) >= 0)
		unit_ut = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) >= 0)
		unit_ut = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) >= 0)
		unit_ut = 2;
	else return (-2);	
	
	Fmt(cmd,"%s<CALC:TCO:U0?");  
    if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
  	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3);
  	
	Scan(strv1,"%s>%f",voltage_u0);
	
	if (FindPattern (stru1, 0, -1, "UV", 0, 0) >= 0)
		unit_u0 = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) >= 0)
		unit_u0 = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) >= 0)
		unit_u0 = 2;
	else return (-3);	

	return (0);
}

/* ========================================================================= */
/*  This function sets the configuration of TC  058 059 060                  */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigTc (int instr_ID, int unit, int typ, int scale)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
	if (bp4462_invalid_integer_range (typ, 0, 13, -2) != 0)
	 	return bp4462_err;
	if (bp4462_invalid_integer_range (scale, 0, 1, -3) != 0)
	 	return bp4462_err;
	if (bp4462_invalid_integer_range (unit, 5, 7, -4) != 0)
	 	return bp4462_err;
	 
    if (bp4462_send_scpi (instr_ID, tc[typ]) != 0)
      	return bp4462_err;
    if (bp4462_send_scpi (instr_ID, tc_scale[scale]) != 0)
      	return bp4462_err;
    if (bp4462_send_scpi (instr_ID, tc_unit[unit-5]) != 0)
      	return bp4462_err;
      
    return(0);
}
/* ========================================================================= */
/*  This function sets only the TC unit 			060         			 */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigTcUnit(int instr_ID, int unit)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
	if (bp4462_invalid_integer_range (unit, 5, 7, -2) != 0)
	 	return bp4462_err;
	 
    if (bp4462_send_scpi (instr_ID, tc_unit[unit-5]) != 0)
      	return bp4462_err;
      
    return(0);
}

/* ========================================================================= */
/*  This function gets the configuration of the TC  058 059 060              */
/* ========================================================================= */
int _VI_FUNC bp4462_GetConfigTc (int instr_ID, int* unit, int* typ, int* scale)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<conf:temp:tco?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
     	
	if(FindPattern (buf, 0, -1, "R", 0, 0) == 0 )
		*typ = 0;
	else if(FindPattern (buf, 0, -1, "S", 0, 0) == 0 )
		*typ = 1;
	else if(FindPattern (buf, 0, -1, "B", 0, 0) == 0 )
		*typ = 2;
	else if(FindPattern (buf, 0, -1, "J", 0, 0) == 0 )
		*typ = 3;
	else if(FindPattern (buf, 0, -1, "T", 0, 0) == 0 )
		*typ = 4;
	else if(FindPattern (buf, 0, -1, "E", 0, 0) == 0 )
		*typ = 5;
	else if(FindPattern (buf, 0, -1, "K", 0, 0) == 0 )
		*typ = 6;
	else if(FindPattern (buf, 0, -1, "U", 0, 0) == 0 )
		*typ = 7;
	else if(FindPattern (buf, 0, -1, "L", 0, 0) == 0 )
		*typ = 8;
	else if(FindPattern (buf, 0, -1, "N", 0, 0) == 0 )
		*typ = 9;
	else if(FindPattern (buf, 0, -1, "C", 0, 0) == 0 )
		*typ = 10;
	else if(FindPattern (buf, 0, -1, "D", 0, 0) == 0 )
		*typ = 11;
	else if(FindPattern (buf, 0, -1, "G2", 0, 0) == 0 )
		*typ = 12;
	else if(FindPattern (buf, 0, -1, "M", 0, 0) == 0 )
		*typ = 13;
	else return (-2); 	
    
    Fmt(cmd,"%s<conf:temp:scal?");
    
    if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
	if(FindPattern (buf, 0, -1, "IPTS68", 0, 0) >= 0 )
		*scale = 0;
	else if(FindPattern (buf, 0, -1, "ITS90", 0, 0) >= 0 )
		*scale = 1;
	else return (-3);	
	
	Fmt(cmd,"%s<unit:temp:tco?");
	
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
     	
	if(FindPattern (buf, 0, -1, "C", 0, 0) >= 0)
		*unit = 5;
	else if(FindPattern (buf, 0, -1, "F", 0, 0) >= 0)
		*unit = 6;
	else if(FindPattern (buf, 0, -1, "K", 0, 0) >= 0)
		*unit = 7;
	else return (-4);	
	
	return(0);
}

/* ========================================================================= */
/*  This function checks the TC-option                                       */
/* ========================================================================= */
int _VI_FUNC bp4462_CheckTcOption (int instr_ID, int* result)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if(bp4462_receive_scpi(instr_ID, "INST:CAT?", buf) != 0)
     	return bp4462_err;
     	
    Fmt (result, "%d<%s", buf); 	

	return (0);
}

/* ========================================================================= */
/*  This function sets the values for the PT100 coefficient 061              */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigCoefficient (int instr_ID, double r0, double a, 
											double b, double c, double t100)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
			 
    Fmt (cmd, "%s<scal:pt100 %f,%f,%f,%f,%f", r0, a, b, c, t100);   
    
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
      	return bp4462_err;
   
    return(0);
}

/*=========================================================================*/
/* This function gets the values of the PT100 coefficient   061            */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigCoefficient (int instr_ID, double* r0, double* a, 
                     						double* b, double* c, double* t100)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<scal:pt100?");
    
    if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;

   	bp4462_valuesplit (buf, r0, a, b, c, t100);		
	
	return(0);
}

/*=========================================================================*/
/* This function gets the DIN-EN values for the PT100 coefficient 062      */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigDinCoefficient (int instr_ID, double* r0, double* a, 
											double* b, double* c, double* t100)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<scal:pt100:din?");
    
    if(bp4462_receive_scpi (instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
	bp4462_valuesplit (buf, r0, a, b, c, t100);
    
	return(0);
}

/* ========================================================================= */
/*  This function sets the configuration of the Voltage output limit  031    */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigOutputLimitV (int instr_ID, int v_limit, int version)
{
  	int maxLimitV;
  	
  	if (version == 1)   // 60 V device version
  		maxLimitV = bp4462_MAX60_limit_V;
  	else				// 30 V device version
 		maxLimitV = bp4462_MAX30_limit_V;

  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	if (bp4462_invalid_integer_range (v_limit, bp4462_MIN_limit_V, maxLimitV, -2) != 0)
    	return bp4462_err;
    
  	Fmt (cmd, "%s<SOUR:VOLT:PROT:LEV %iV", v_limit);
  
  	if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;
    
  	return(0);
}

/* ========================================================================= */
/*  This function sets the configuration of the Current output limit   031   */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigOutputLimitC (int instr_ID, int c_limit, int version)
{
  	int maxLimitC;
  	
  	if (version == 1)   // 60 V device version
  		maxLimitC = bp4462_MAX60_limit_MA;
  	else				// 30 V device version
 		maxLimitC = bp4462_MAX30_limit_MA;

  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
    
  	if (bp4462_invalid_integer_range (c_limit, bp4462_MIN_limit_MA, maxLimitC, -2) != 0)
    	return bp4462_err;
	
  	Fmt (cmd, "%s<sour:curr:prot:lev %iMA", c_limit);
  	
  	if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;
    
  	return(0);
}

/*============================================================================*/
/* This function gets the configuration of the Volt and Curr output limit 031 */
/*============================================================================*/
int _VI_FUNC bp4462_GetConfigOutputLimit (int instr_ID, int* v_limit, int* c_limit)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	Fmt (cmd,"%s<sour:volt:prot:lev?");
  	if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
  	Scan (buf,"%s>%i", v_limit);
    
  	Fmt (cmd,"%s<sour:curr:prot:lev?");
  	if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
  	Scan (buf,"%s>%i", c_limit);
	
  	return(0);
}

/* ========================================================================= */
/*  This function sets output status an value (Volt/Curr/Temp)  016 017 018  */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigOutputStatus (int instr_ID, int unit, double value, int version)
{
  	char unit_char[10]; 
  	int minVolt, maxVolt, minCurrUA, maxCurrUA, minCurrMA, maxCurrMA;
  	
  	if (version == 1)			// 60 Volt device version
  	{
  		minVolt = bp4462_MIN60_V;
  		maxVolt = bp4462_MAX60_V;
  		minCurrUA = bp4462_MIN60_UA;
  		maxCurrUA = bp4462_MAX60_UA;
  		minCurrMA = bp4462_MIN60_MA;
  		maxCurrMA = bp4462_MAX60_MA;
  	}
  	else						// 30 Volt device version 
  	{
  		minVolt = bp4462_MIN30_V;
    	maxVolt = bp4462_MAX30_V;
  		minCurrUA = bp4462_MIN30_UA;
  		maxCurrUA = bp4462_MAX30_UA;
  		minCurrMA = bp4462_MIN30_MA;
  		maxCurrMA = bp4462_MAX30_MA;
    }

  	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	if (bp4462_invalid_integer_range (unit, 0, 7, -2) != 0)
    	return bp4462_err;
	 
  	if ((unit >= 0) && (unit <= 2)) 		/* Voltage*/
		{
    	if (unit == 0) 						/* V */
      		{
	  		Fmt (unit_char,"%s<UV");
      		if (bp4462_invalid_real_range (value, bp4462_MIN_UV, bp4462_MAX_UV,  -3) != 0)
        		return bp4462_err;
      		}
    	else if (unit == 1) 				/* mV */
      		{
	  		Fmt(unit_char,"%s<MV");
      		if (bp4462_invalid_real_range (value, bp4462_MIN_MV, bp4462_MAX_MV,  -3) != 0)
        		return bp4462_err;
      		}
     	else								/* V */
      		{
      		Fmt(unit_char,"%s<V");
      		if (bp4462_invalid_real_range (value, minVolt, maxVolt,  -3) != 0)
        		return bp4462_err;
      		}  
      		
    	Fmt (cmd, "%s<SOUR:VOLT %f%s",value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	}
  	else if ((unit == 3)||(unit == 4))  	/* Current */
    	{
    	if (unit == 3) 						/* A */
      		{
	  		Fmt (unit_char,"%s<UA");
      		if (bp4462_invalid_real_range (value, minCurrUA, maxCurrUA,  -4) != 0)
        		return bp4462_err;
      		}
    	else 				 				/* mA */
      		{
	  		Fmt(unit_char,"%s<MA");
      		if (bp4462_invalid_real_range (value, minCurrMA, maxCurrMA,  -4) != 0)
        		return bp4462_err;
      		}
  
    	Fmt (cmd, "%s<SOUR:CURR %f%s",value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	}
  	else if ((unit >= 5) && (unit <= 7)) 	/* TC*/
    	{
		switch (unit)
	  		{
	  		case 5:		/* C*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_GRD_C, bp4462_MAX_GRD_C,  -5) != 0)
          			return bp4462_err;
	    		break;
	  		case 6:		/* F*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_GRD_F, bp4462_MAX_GRD_F,  -5) != 0)
          			return bp4462_err;
	    		break;
	  		case 7:		/* K*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_K, bp4462_MAX_K,  -5) != 0)
          			return bp4462_err;
	    		break;
      		}  
    
    	Fmt (cmd, "%s<SOUR:TCO %f%s",value, unit_type[unit]);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;

    	}
  	return(0);
}

/* ========================================================================= */
/*  This function gets output status  (Volt/Curr/Temp)                       */
/* ========================================================================= */
int _VI_FUNC bp4462_GetConfigOutputStatus (int instr_ID, int* unit, double* value) 
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt(cmd,"%s<sour:volt?");				 
    if(bp4462_receive_scpi (instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     
  	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3);
  	  
  	Scan (strv1, "%s>%f", value); 
  	
	if (FindPattern (buf, 0, -1, "UV", 0, 0) != -1)
		*unit = 0;
	else if (FindPattern (buf, 0, -1, "MV", 0, 0) != -1)
		*unit = 1;
	else if (FindPattern (buf, 0, -1, "V", 0, 0) != -1)
		*unit = 2;
	else if (FindPattern (buf, 0, -1, "UA", 0, 0) != -1)
		*unit = 3;
	else if (FindPattern (buf, 0, -1, "MA", 0, 0) != -1)
		*unit = 4;
	else if (FindPattern (buf, 0, -1, "C", 0, 0) != -1)
		*unit = 5;
	else if (FindPattern (buf, 0, -1, "F", 0, 0) != -1)
		*unit = 6;
	else if (FindPattern (buf, 0, -1, "K", 0, 0) != -1)
		*unit = 7;
	else return (-2);	
	
	return(0);
}

/*=========================================================================*/
/* This function sets the configuration of the output mode      032        */
/*=========================================================================*/
int _VI_FUNC bp4462_SetConfigOutputMode (int instr_ID, int mode)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_invalid_integer_range (mode, 0, 2,  -2) != 0)
    	return bp4462_err;

  	if (bp4462_send_scpi (instr_ID, output_mode[mode]) != 0)
    	return bp4462_err;

  	return (0);
}

/*=========================================================================*/
/* This function gets the configuration of the output mode     032         */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigOutputMode(int instr_ID, int *mode)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	Fmt(cmd,"%s<sour:mode?");
  	if(bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;
    
  	if(FindPattern (buf, 0, -1, "FIX", 0, 0) >= 0)
		*mode = 0;
  	else if(FindPattern (buf, 0, -1, "SWE", 0, 0) >= 0)
		*mode = 1;
  	else if(FindPattern (buf, 0, -1, "LIST", 0, 0) >= 0)
		*mode = 2;
	else return (-2);	

  	return (0);
}
		   
/*=========================================================================*/
/* This function sets the sense                             030            */
/*=========================================================================*/
int _VI_FUNC bp4462_SetConfigSense (int instr_ID, int sense)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_invalid_integer_range (sense, 0, 1,  -2) != 0)
    	return bp4462_err;

  	if (bp4462_send_scpi (instr_ID, sense_mode[sense]) != 0)
    	return bp4462_err;

  	return (0);
}

/*=========================================================================*/
/* This function gets the sense-mode                          030          */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigSense (int instr_ID, int* sense)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	Fmt (cmd,"%s<sour:volt:alc:sour?");
  	if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
    	return bp4462_err;

  	if (FindPattern (buf, 0, -1, "INT", 0, 0) >= 0)
		*sense = 0;
  	else if (FindPattern (buf, 0, -1, "EXT", 0, 0) >= 0)
		*sense = 1;
  	else return (-2);		
  	
  	return (0);
}

/* ========================================================================= */
/*  This function sets the configuration of the Display mode  065            */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigDisplayMode (int instr_ID, int d_mode)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
	if (bp4462_invalid_integer_range (d_mode, 0, 5, -2) != 0)
	 	return bp4462_err;
	 
    if (bp4462_send_scpi (instr_ID, display_mode[d_mode]) != 0)
      	return bp4462_err;
      
    return(0);
}

/*=========================================================================*/
/* This function set the display contrast   064                            */
/*=========================================================================*/
int _VI_FUNC bp4462_SetConfigDisplayContrast(int instr_ID, double contrast)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
    
    if (bp4462_invalid_real_range (contrast, 0.0, 1.0,  -2) != 0)
      	return bp4462_err;

    Fmt (buf,"%s<:DISP:CONT %f",contrast);         /* Display contrast*/
    if (bp4462_send_scpi(instr_ID, buf) != 0)
     	return bp4462_err;

    return (0);
}
/*=========================================================================*/
/* This function get the display contrast    064                           */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigDisplayContrast (int instr_ID, double* contrast)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    Fmt (cmd,"%s<:DISP:CONT?");         		/* Display contrast*/
    
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
    Scan (buf,"%s>%f", contrast);			     

    return (0);
}

/*===================================================================================================================================*/
/* This function set the device access for the menue 067 068 069 070 071 072 073 074 075 076 077 078 079 080 081 082 083 084 085 086 */
/*===================================================================================================================================*/
int _VI_FUNC bp4462_SetConfigDeviceAccess (int instr_ID, int speech, int output_m,
								int ramp_mode, int ramp1, int ramp2, int delta_v,
								int store_config, int recall_config, int sense, 
								int output_limit,int range_mode, int divide, 
								int TC_config,int RJ_config,int PT100_scale,
								int device_store, int device_recall,int interf,
								int contrast,int status_display)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
    
    if (bp4462_invalid_integer_range (speech, 0, 1,  -2) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (output_m, 0, 1,  -3) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (ramp_mode, 0, 1,  -4) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (ramp1, 0, 1,  -5) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (ramp2, 0, 1,  -6) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (delta_v, 0, 1,  -7) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (store_config, 0, 1,  -8) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (recall_config, 0, 1,  -9) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (sense, 0, 1,  -10) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (output_limit, 0, 1,  -11) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (range_mode, 0, 1,  -12) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (divide, 0, 1,  -13) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (TC_config, 0, 1,  -14) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (RJ_config, 0, 1,  -15) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (PT100_scale, 0, 1,  -16) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (device_store, 0, 1,  -17) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (device_recall, 0, 1,  -18) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (interf, 0, 1,  -19) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (contrast, 0, 1,  -20) != 0)
      	return bp4462_err;
    if (bp4462_invalid_integer_range (status_display, 0, 1,  -21) != 0)
      	return bp4462_err;
    
    Fmt(buf,"%s<acc:lang %d",speech);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;			  
    Fmt(buf,"%s<acc:sour %d",output_m);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:ramp %d",ramp_mode);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:swe:conf %d",ramp1);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:list:conf %d",ramp2);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:delt %d",delta_v);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:stor %d",store_config);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:rec %d",recall_config);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:sens %d",sense);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:prot %d",output_limit);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:rang %d",range_mode);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:div %d",divide);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;			 
    Fmt(buf,"%s<acc:tco %d",TC_config);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:refj %d",RJ_config);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:scal %d",PT100_scale);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:conf:stor %d",device_store);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:conf:rec %d",device_recall);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:comm %d",interf);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:cont %d",contrast);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
    Fmt(buf,"%s<acc:stat %d",status_display);
    if(bp4462_send_scpi(instr_ID, buf) != 0) 
     	return bp4462_err;
     	  
    return(0);
}

/*=========================================================================*/
/* This function get the device access for the menue 067 068 069 070 071 072 073 074 075 076 077 078 079 080 081 082 083 084 085 086 */
/*=========================================================================*/
int _VI_FUNC bp4462_GetConfigDeviceAccess (int instr_ID, int* speech, int* output_m, 
								int* ramp_mode, int* ramp1, int* ramp2, int* delta_v, 
								int* store_config, int* recall_config,int* sense,
								int* output_limit, int* range_mode, int* divide,
								int* TC_config, int* RJ_config, int* PT100_scale, 
								int* device_store, int* device_recall, int* interf,
								int* contrast, int* status_display)
{
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
    		   
    Fmt(cmd,"%s<acc:lang?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",speech);
    
    Fmt(cmd,"%s<acc:sour?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",output_m);
    
    Fmt(cmd,"%s<acc:ramp?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",ramp_mode);
    
    Fmt(cmd,"%s<acc:swe:conf?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",ramp1);
    
    Fmt(cmd,"%s<acc:list:conf?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",ramp2);
    
    Fmt(cmd,"%s<acc:delt?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",delta_v);
    
    Fmt(cmd,"%s<acc:stor?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",store_config);
    
    Fmt(cmd,"%s<acc:rec?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",recall_config);
    
    Fmt(cmd,"%s<acc:sens?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",sense);
    
    Fmt(cmd,"%s<acc:prot?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",output_limit);
    
    Fmt(cmd,"%s<acc:rang?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",range_mode);
    
    Fmt(cmd,"%s<acc:div?");						   
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",divide);
    
    Fmt(cmd,"%s<acc:tco?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",TC_config);
    
    Fmt(cmd,"%s<acc:refj?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",RJ_config);
    
    Fmt(cmd,"%s<acc:scal?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",PT100_scale);
    
    Fmt(cmd,"%s<acc:conf:stor?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",device_store);
    
    Fmt(cmd,"%s<acc:conf:rec?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",device_recall);
    
    Fmt(cmd,"%s<acc:comm?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",interf);
    
    Fmt(cmd,"%s<acc:cont?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",contrast);
    
    Fmt(cmd,"%s<acc:stat?");
    if(bp4462_receive_scpi(instr_ID, cmd,buf) != 0)
     	return bp4462_err;
	Scan(buf,"%s>%i",status_display);
    
    return(0);
}

/* ========================================================================= */
/*  This function sets the configuration of the voltage split 052            */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigVoltageDiv (int instr_ID, double factor)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
	if (bp4462_invalid_real_range (factor, 1.0, 1010.0, -2) != 0)
	 	return bp4462_err;
 	
    Fmt (cmd, "%s<SOUR:VOLT:DIV %f",factor);
    
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
      	return bp4462_err;
	
	return 0;
}

/* ========================================================================= */
/*  This function sets the configuration of the voltage split state 052      */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigVoltageDivState (int instr_ID, int state)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
	if (bp4462_invalid_integer_range (state, 0, 1, -2) != 0)
	 	return bp4462_err;
	 
    Fmt (cmd, "%s<SOUR:VOLT:DIV:STAT %d", state);
    if (bp4462_send_scpi (instr_ID, cmd) != 0)
      	return bp4462_err;

	return (0);
}	

/* ========================================================================= */
/*  This function gets the configuration of the voltage split state 052      */
/* ========================================================================= */
int _VI_FUNC bp4462_GetConfigVoltageDivState(int instr_ID, int* mode, double* factor)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	
    Fmt (cmd,"%s<sour:volt:div?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
	Scan (buf,"%s>%f", factor);
    
    Fmt (cmd,"%s<sour:volt:div:stat?");
    if (bp4462_receive_scpi(instr_ID, cmd, buf) != 0)
     	return bp4462_err;
     	
	Scan (buf,"%s>%i", mode);
	
	return (0);
}

/* ========================================================================= */
/*  This function sets delta values   (Volt/Curr/Temp)  021 022 023          */
/* ========================================================================= */
int _VI_FUNC bp4462_SetConfigDeltaValue (int instr_ID, int unit, double value, int version)
{
  	char unit_char[10]; 
  	int minUVolt, maxUVolt, minMVolt, maxMVolt, minVolt, maxVolt, minCurrUA, maxCurrUA, minCurrMA, maxCurrMA;
  	
  	if (version == 1)			// 60 Volt device version
  	{
  		minUVolt = bp4462_MIN60_d_UV;
  		maxUVolt = bp4462_MAX60_d_UV;
  		minMVolt = bp4462_MIN60_d_MV;
  		maxMVolt = bp4462_MAX60_d_MV;
  		minVolt = bp4462_MIN60_d_V;
  		maxVolt = bp4462_MAX60_d_V; 
  		minCurrUA = bp4462_MIN60_d_UA;
  		maxCurrUA = bp4462_MAX60_d_UA;
  		minCurrMA = bp4462_MIN60_d_MA;
  		maxCurrMA = bp4462_MAX60_d_MA;  
  	}
  	else						// 30 Volt device version 
  	{
  		minUVolt = bp4462_MIN30_d_UV;
  		maxUVolt = bp4462_MAX30_d_UV;
  		minMVolt = bp4462_MIN30_d_MV;
  		maxMVolt = bp4462_MAX30_d_MV;
  		minVolt = bp4462_MIN30_d_V;
  		maxVolt = bp4462_MAX30_d_V; 
  		minCurrUA = bp4462_MIN30_d_UA;
  		maxCurrUA = bp4462_MAX30_d_UA;
  		minCurrMA = bp4462_MIN30_d_MA;
  		maxCurrMA = bp4462_MAX30_d_MA;  
    }

  	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
			  
  	if (bp4462_invalid_integer_range (unit, 0, 7,  -2) != 0)
    	return bp4462_err;
		   
  	if ((unit >= 0) && (unit <= 2)) 	/* Voltage */
		{
    	if (unit == 0) 					/* V */
      		{
	  		Fmt (unit_char,"%s<UV");
      		if (bp4462_invalid_real_range (value, minUVolt, maxUVolt,  -3) != 0)
        		return bp4462_err;
      		}
    	else if (unit == 1) 			/* mV */
      		{
	  		Fmt (unit_char,"%s<MV");
      		if (bp4462_invalid_real_range (value, minMVolt, maxMVolt,  -3) != 0)
        		return bp4462_err;
      		}
     	else 							/* V */
      		{
      		Fmt (unit_char,"%s<V");
      		if (bp4462_invalid_real_range (value, minVolt, maxVolt,  -3) != 0)
        		return bp4462_err;
      		}  
      		
   	Fmt (cmd, "%s<SOUR:VOLT:DELT %f%s", value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	}
  	else if ((unit == 3)||(unit == 4))  /* Current */
    	{
    	if (unit == 3) 					/* A */
      		{
	  		Fmt (unit_char,"%s<UA");
      		if (bp4462_invalid_real_range (value, minCurrUA, maxCurrUA,  -4) != 0)
        		return bp4462_err;
      		}
    	else if (unit == 4) 			/* mA */
      		{
	  		Fmt(unit_char,"%s<MA");
      		if (bp4462_invalid_real_range (value, minCurrMA, maxCurrMA,  -4) != 0)
        		return bp4462_err;
      		}

    	Fmt (cmd, "%s<SOUR:CURR:DELT %f%s", value, unit_char);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
    	}
  	else if ((unit >= 5) && (unit <= 7)) /* TC*/
    	{
		switch (unit)
	  		{
	  		case 5:		/* C*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_d_GRD_C, bp4462_MAX_d_GRD_C,  -5) != 0)
          			return bp4462_err;
	    		break;
	  		case 6:		/* F*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_d_GRD_F, bp4462_MAX_d_GRD_F,  -5) != 0)
          			return bp4462_err;
	    		break;
	  		case 7:		/* K*/
        		if (bp4462_invalid_real_range (value, bp4462_MIN_d_K, bp4462_MAX_d_K,  -5) != 0)
          			return bp4462_err;
	    		break;
      		}  
      		
  	Fmt (cmd, "%s<SOUR:TCO:DELT %f%s",value, unit_type[unit]);
    	if (bp4462_send_scpi (instr_ID, cmd) != 0)
      		return bp4462_err;
		}
    
  	return (0);
}

/* ========================================================================= */
/*  This function gets delta value  (Volt/Curr/Temp)     021 022 023         */
/* ========================================================================= */
int _VI_FUNC bp4462_GetConfigDeltaValue (int instr_ID, int type, int *unit, 
										double *value) 
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
	
  	if (bp4462_invalid_integer_range (type, 0, 7,  -2) != 0)
    	return bp4462_err;
	 
  	if (bp4462_receive_scpi (instr_ID, delta_value[type], buf) != 0)
    	return bp4462_err;
    
  	bp4462_strsplit(buf, strv1, stru1, strv2, stru2, strv3, stru3); 
 	 
	if (FindPattern (stru1, 0, -1, "UV", 0, 0) != -1)
		*unit = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) != -1)
		*unit = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) != -1)
		*unit = 2;
	else if (FindPattern (stru1, 0, -1, "UA", 0, 0) != -1)
		*unit = 3;
	else if (FindPattern (stru1, 0, -1, "MA", 0, 0) != -1)
		*unit = 4;
	else if (FindPattern (stru1, 0, -1, "C", 0, 0) != -1)
		*unit = 5;
	else if (FindPattern (stru1, 0, -1, "F", 0, 0) != -1)
		*unit = 6;
	else if (FindPattern (stru1, 0, -1, "K", 0, 0) != -1)
		*unit = 7;
	else return (-3);	

  	Scan(strv1, "%s>%f", value); 					  
	
  	return(0);
}

/*=========================================================================*/
/* This function add/sub a delta-value to the output value      024        */
/*=========================================================================*/
int _VI_FUNC bp4462_SetAddDeltaValue (int instr_ID, int mode)
{
  bp4462_err = 0;
	
  if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    return bp4462_err;

  if (bp4462_invalid_integer_range (mode, 0, 1,  -2) != 0)
    return bp4462_err;

  if (bp4462_send_scpi (instr_ID, set_delta[mode]) != 0)
    return bp4462_err;

  return (0);
}


/*=========================================================================*/
/* This function sends SCPI                             006                */
/*=========================================================================*/
int _VI_FUNC  bp4462_send_scpi(int instr_ID, char* send_buffer)
{
   	bp4462_err = 0;
   
   	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

   	Fmt(cmd,"%s<%s\n",send_buffer);
		
   	if(interface == bp4462_GPIB)  	/* GPIB ?? */
   		{
    	bp4462_err = bp4462_write_msg_gpib(instr_ID, cmd, NumFmtdBytes ());

    	if(bp4462_err != 0)
     		return (bp4462_err);
   		}
   	else    						/* Serial */
   		{
    	bp4462_err = bp4462_send_rs232(instr_ID, cmd);
    	if(bp4462_err != 0)
     		return (bp4462_err);
   		}
   	return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC  bp4462_receive_scpi(int instr_ID, char* send_buffer,char* receive_buffer)
{
   
   	bp4462_err = 0;
   
   	Fmt(cmd,"%s<%s\n",send_buffer);
   
   	if(interface == bp4462_GPIB)  		/* GPIB ?? */
   		{
	    bp4462_err = bp4462_write_msg_gpib(instr_ID, cmd,NumFmtdBytes ());
	    if(bp4462_err != 0)
     		return (bp4462_err);

    	bp4462_err = bp4462_read_msg_gpib(instr_ID, receive_buffer,bp4462_MAX_CMD);
    	if(bp4462_err != 0)
     		return (bp4462_err);
   		}
   	else								/* Serial */
   		{
	    bp4462_err = bp4462_receive_rs232(instr_ID, cmd, receive_buffer);
	    if(bp4462_err != 0)
     		return (bp4462_err);
   		}
   
   	return(0);
}

/* ========================================================================= */
/*  This function sets the lock state for the device keyboard        014     */
/* ========================================================================= */
int _VI_FUNC  bp4462_SetLockSystemKeyboard (int instr_ID, int lock_mode)
{
  	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_invalid_integer_range (lock_mode, 0, 1,  -2) != 0)
    	return bp4462_err;

  	Fmt(buf,"%s<syst:kloc %d",lock_mode);
  	if (bp4462_send_scpi (instr_ID, buf) != 0)
    	return bp4462_err;

  	return 0;
}

/* ========================================================================= */
/*  This function get lock mode of the device keyboard               014     */
/* ========================================================================= */
int _VI_FUNC  bp4462_GetLockSystemKeyboard (int instr_ID, int *lock_mode)
{
 	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_receive_scpi(instr_ID, "SYST:KLOC?", buf) != 0)
    	return (bp4462_err);
  
  	Scan (buf,"%s>%i", lock_mode); 

  	return 0;
}

/*=========================================================================*/
/* This function get system error                                   013    */
/*=========================================================================*/
int _VI_FUNC  bp4462_GetSystemError (int instr_ID)
{
	int err, ret_error, found_error;
	
	bp4462_err = 0;
   
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	err = -1;
  	ret_error = 0;
  	found_error = 0;

  	while (err != 0) 			/* Einlesen bis kein Fehler kommt */
    	{
    	if (bp4462_receive_scpi (instr_ID, "SYSTEM:ERROR?", buf) != 0)
      		return (bp4462_err);

    	Scan (buf, "%s>%i", &err);
	/*  FmtOut("\nSys %s",buf); */
    	if((err != 0) && (found_error == 0))
      		{
      		found_error = 1;    /* Save only the first Error */
      		ret_error = err;
      		}
    	}
   
  	return (ret_error);
}

/*=============================================================================*/
/* This function stores the output- and the delta-X-value at a selected No 025 */
/*=============================================================================*/
int _VI_FUNC bp4462_SetStoreOutputValue (int instr_ID, int store_id)
{
  	bp4462_err = 0;
	
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_invalid_integer_range (store_id, 0, 99,  -2) != 0)
    	return bp4462_err;

  	Fmt (buf, "%s<sour:stor %d", store_id);        
  	if (bp4462_send_scpi (instr_ID, buf) != 0)
    	return bp4462_err;
	
  	return (0);
}
		 
/*==============================================================================*/
/* This function gets the output- and the delta-X-value stored at a selected ID */
/*==============================================================================*/
int _VI_FUNC bp4462_GetStoreOutputValue (int instr_ID, int store_id, int* out_unit, 
										double* out_value, int* delta_unit, 
										double* delta_value)
{
	bp4462_err = 0;
	
	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;
	 
	if (bp4462_invalid_integer_range (store_id, 0, 99, -2) != 0)
     	return bp4462_err;
	 
    Fmt (cmd, "%s<sour:stor? %d", store_id);
    if (bp4462_receive_scpi (instr_ID, cmd, buf) != 0)
     	return bp4462_err;
	
	bp4462_strsplit (buf, strv1, stru1, strv2, stru2, strv3, stru3);
	
	if (FindPattern (stru1, 0, -1, "UV", 0, 0) != -1)
		*out_unit = 0;
	else if (FindPattern (stru1, 0, -1, "MV", 0, 0) != -1)
		*out_unit = 1;
	else if (FindPattern (stru1, 0, -1, "V", 0, 0) != -1)
		*out_unit = 2;
	else if (FindPattern (stru1, 0, -1, "UA", 0, 0) != -1)
		*out_unit = 3;
	else if (FindPattern (stru1, 0, -1, "MA", 0, 0) != -1)
		*out_unit = 4;
	else if (FindPattern (stru1, 0, -1, "C", 0, 0) != -1)
		*out_unit = 5;
	else if (FindPattern (stru1, 0, -1, "F", 0, 0) != -1)
		*out_unit = 6;
	else if (FindPattern (stru1, 0, -1, "K", 0, 0) != -1)
		*out_unit = 7;
	else return (-3);	

	if (FindPattern (stru2, 0, -1, "UV", 0, 0) != -1)
		*delta_unit = 0;
	else if (FindPattern (stru2, 0, -1, "MV", 0, 0) != -1)
		*delta_unit = 1;
	else if (FindPattern (stru2, 0, -1, "V", 0, 0) != -1)
		*delta_unit = 2;
	else if (FindPattern (stru2, 0, -1, "UA", 0, 0) != -1)
		*delta_unit = 3;
	else if (FindPattern (stru2, 0, -1, "MA", 0, 0) != -1)
		*delta_unit = 4;
	else if (FindPattern (stru2, 0, -1, "C", 0, 0) != -1)
		*delta_unit = 5;
	else if (FindPattern (stru2, 0, -1, "F", 0, 0) != -1)
		*delta_unit = 6;
	else if (FindPattern (stru2, 0, -1, "K", 0, 0) != -1)
		*delta_unit = 7;
	else return (-4);	

  	Scan (strv1, "%s>%f", out_value); 
	Scan (strv2, "%s>%f", delta_value);
	
    return (0);
}

/* ========================================================================= */
/*  This function store output values   (V/C/TC)      027 028 029            */
/*  Changes: Added the 'version' parameter for the 60 Volt 4462 Device 			*/
/*  		in Release V 2.0   21.05.2002 by T. Meder							*/
/* ========================================================================= */
int _VI_FUNC bp4462_SetStoreIdOutputValue ( int instr_ID, int number, 
											int output_unit, double output_value, 
											int delta_unit, double delta_value, int version)
{
  	char output_unit_char[10]; 
  	char delta_unit_char[10];
  	int minVolt, maxVolt, minCurrUA, maxCurrUA, minCurrMA, maxCurrMA;
  	
  	if (version == 1)			// 60 Volt device version
  	{
  		minVolt = bp4462_MIN60_V;
  		maxVolt = bp4462_MAX60_V;
  		minCurrUA = bp4462_MIN60_UA;
  		maxCurrUA = bp4462_MAX60_UA;
  		minCurrMA = bp4462_MIN60_MA;
  		maxCurrMA = bp4462_MAX60_MA;
  	}
  	else						// 30 Volt device version 
  	{
  		minVolt = bp4462_MIN30_V;
    	maxVolt = bp4462_MAX30_V;
  		minCurrUA = bp4462_MIN30_UA;
  		maxCurrUA = bp4462_MAX30_UA;
  		minCurrMA = bp4462_MIN30_MA;
  		maxCurrMA = bp4462_MAX30_MA;
    }
	
  	bp4462_err = 0;
    
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;
    
  	if (bp4462_invalid_integer_range (number, 0, 99,  -2) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (output_unit, 0, 7,  -3) != 0)
    	return bp4462_err;
  	if (bp4462_invalid_integer_range (delta_unit, 0, 7,  -4) != 0)
    	return bp4462_err;
    
  	if ((output_unit >= 0) && (output_unit <= 2)) 		/* Voltage*/
		{
    	if (output_unit == 0) 							/* V */
      		{
	  		Fmt (output_unit_char,"%s<UV");
      		if (bp4462_invalid_real_range (output_value, bp4462_MIN_UV, bp4462_MAX_UV,  -5) != 0)
        		return bp4462_err;
      		}
    	else if (output_unit == 1) 						/* mV */
      		{
	  		Fmt(output_unit_char,"%s<MV");
      		if (bp4462_invalid_real_range (output_value, bp4462_MIN_MV, bp4462_MAX_MV,  -5) != 0)
        		return bp4462_err;
      		}
     	else											/* V */
      		{
      		Fmt(output_unit_char,"%s<V");
      		if (bp4462_invalid_real_range (output_value, minVolt, maxVolt,  -5) != 0)
        		return bp4462_err;
      		}  
   		}
   		
  	else if ((output_unit == 3)||(output_unit == 4))   	/* Current */
    	{
    	if (output_unit == 3) 							/* A */
      		{
	  		Fmt (output_unit_char,"%s<UA");
      		if (bp4462_invalid_real_range (output_value, minCurrUA, maxCurrUA,  -6) != 0)
        		return bp4462_err;
      		}
    	else if (output_unit == 4) 						/* mA */
      		{
	  		Fmt(output_unit_char,"%s<MA");
      		if (bp4462_invalid_real_range (output_value, minCurrMA, maxCurrMA,  -6) != 0)
        		return bp4462_err;
      		}
    	}
    	
  	else if ((output_unit >= 5) && (output_unit <= 7)) 	/* TC*/
    	{
		switch(output_unit)
	  		{
	  		case 5:		/* C*/
	  			Fmt (output_unit_char,"%s<C");
        		if (bp4462_invalid_real_range (output_value, bp4462_MIN_GRD_C, bp4462_MAX_GRD_C,  -7) != 0)
          			return bp4462_err;
	    		break;
	  		case 6:		/* F*/
	  			Fmt (output_unit_char,"%s<F");
        		if (bp4462_invalid_real_range (output_value, bp4462_MIN_GRD_F, bp4462_MAX_GRD_F,  -7) != 0)
          			return bp4462_err;
	    		break;
	  		case 7:		/* K*/
	  			Fmt (output_unit_char,"%s<K");
        		if (bp4462_invalid_real_range (output_value, bp4462_MIN_K, bp4462_MAX_K,  -7) != 0)
          			return bp4462_err;
	    		break;
	  		}  
    	}
    /*---- Delta value and unit ----*/
  	if((delta_unit >= 0) && (delta_unit <= 2)) 			/* Voltage*/
		{
    	if (delta_unit == 0) 							/* V */
      		{
	  		Fmt (delta_unit_char,"%s<UV");
      		if (bp4462_invalid_real_range (delta_value, bp4462_MIN_UV, bp4462_MAX_UV,  -8) != 0)
        		return bp4462_err;
      		}
    	else if (delta_unit == 1) 						/* mV */
      		{
	  		Fmt(delta_unit_char,"%s<MV");
      		if (bp4462_invalid_real_range (delta_value, bp4462_MIN_MV, bp4462_MAX_MV,  -8) != 0)
        		return bp4462_err;
      		}
     	else											/* V */
      		{
      		Fmt(delta_unit_char,"%s<V");
      		if (bp4462_invalid_real_range (delta_value, minVolt, maxVolt,  -8) != 0)
        		return bp4462_err;
      		}  
   		}
   		
  	else if ((delta_unit == 3)||(delta_unit == 4))   	/* Current */
    	{
    	if (delta_unit == 3) 							/* A */
      		{
	  		Fmt (delta_unit_char,"%s<UA");
      		if (bp4462_invalid_real_range (delta_value, minCurrUA, maxCurrUA,  -9) != 0)
        		return bp4462_err;
      		}
    	else if (delta_unit == 4) 						/* mA */
      		{
	  		Fmt(delta_unit_char,"%s<MA");
      		if (bp4462_invalid_real_range (delta_value, minCurrMA, maxCurrMA,  -9) != 0)
        		return bp4462_err;
      		}
    	}
    	
  	else if ((delta_unit >= 5) && (delta_unit <= 7)) /* TC*/
    	{
		switch(delta_unit)
	  		{
	  		case 5:		/* C*/
	  			Fmt (delta_unit_char,"%s<C");
        		if (bp4462_invalid_real_range (delta_value, bp4462_MIN_GRD_C, bp4462_MAX_GRD_C,  -10) != 0)
          			return bp4462_err;
	    		break;
	  		case 6:		/* F*/
	  			Fmt (delta_unit_char,"%s<F");
        		if (bp4462_invalid_real_range (delta_value, bp4462_MIN_GRD_F, bp4462_MAX_GRD_F,  -10) != 0)
          			return bp4462_err;
	    		break;
	  		case 7:		/* K*/
	  			Fmt (delta_unit_char,"%s<K");
        		if (bp4462_invalid_real_range (delta_value, bp4462_MIN_K, bp4462_MAX_K,  -10) != 0)
          			return bp4462_err;
	    		break;
	  		}  
    	}
    
  	if (((output_unit >= 0) && (output_unit <= 2)) && ((delta_unit >= 0) && (delta_unit <= 2)))
    	Fmt (cmd, "%s<SOUR:STOR:VOLT %d,%f%s,%f%s",number, output_value, output_unit_char, 
      												delta_value, delta_unit_char);
  	else if (((output_unit >= 3) && (output_unit <= 4)) && ((delta_unit >= 3) && (delta_unit <= 4)))
    	Fmt (cmd, "%s<SOUR:STOR:CURR %d,%f%s,%f%s",number, output_value, output_unit_char, 
      												delta_value, delta_unit_char);
  	else if (((output_unit >= 5) && (output_unit <= 7)) && ((delta_unit >= 5) && (delta_unit <= 7)))
    	Fmt (cmd, "%s<SOUR:STOR:TCO %d,%f%s,%f%s",number, output_value, output_unit_char, 
      												delta_value, delta_unit_char);
  	else return (-11);
    
  	if (bp4462_send_scpi (instr_ID, cmd) != 0)
    	return bp4462_err;
      
  	return(0);   
}

/* ========================================================================= */
/*  This function recall memory label     063                                */
/* ========================================================================= */
int _VI_FUNC bp4462_GetMemoryLabel (int instr_ID, int mem_no, char* label)
{
    int ret;
													  
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (mem_no, 1, 33,  -2) != 0)
     	return bp4462_err;

    Fmt (cmd, "%s<mem:stat:name? %d", mem_no);
    if (bp4462_receive_scpi (instr_ID, cmd, buf) != 0)
      	return bp4462_err;
      	
	Scan (buf, "%s>%s", label);
	
    	return (0);
}

/* ========================================================================= */
/*  This function stores a memory label to a selected memory device ID  063  */
/* ========================================================================= */
int _VI_FUNC bp4462_SetMemoryLabel (int instr_ID, int mem_no, char label)
{
    int ret;
													  
    bp4462_err = 0;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     	return bp4462_err;

    if (bp4462_invalid_integer_range (mem_no, 1, 33,  -2) != 0)
     	return bp4462_err;
     	
    Fmt (buf, "%s<mem:stat:def %s,%d", label, mem_no);
    
    if (bp4462_send_scpi(instr_ID, buf) != 0)
     	return bp4462_err;
     
    	return (0);
}


/***************** Register ****************************************************/

/*=========================================================================*/
/* This function read Output Error Id-No.							   096 */
/*=========================================================================*/
int _VI_FUNC  bp4462_CheckErrorStatusOutput (int instr_ID)
{ 
  int quest_flag;

  bp4462_err = 0;
   
  if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    return bp4462_err;

  if(bp4462_receive_scpi(instr_ID, "S:Q:C?",buf) != 0)
    return (bp4462_err);
   
	Scan (buf, "%s>%i[r16]", &quest_flag);
   	
   if (quest_flag & bp4462_VOLTAGE_ERROR)   	/* Bit 0 of Questionable Status Register */ 
		{
    	bp4462_err = 310;
    	return bp4462_err;
   		}
   if (quest_flag & bp4462_CURRENT_ERROR)   	/* Bit 1 of Questionable Status Register */   
		{
    	bp4462_err = 311;
    	return bp4462_err;
   		}
   if (quest_flag & bp4462_TEMPERATURE_ERROR)	/* Bit 4 of Questionable Status Register */ 
		{
    	bp4462_err = 313;
    	return bp4462_err;
   		}
   if (quest_flag & bp4462_POWER_ERROR)   	    /* Bit 3 of Questionable Status Register */ 
		{
    	bp4462_err = 312;
    	return bp4462_err;
   		}
   if (quest_flag & bp4462_CALIBRATION_ERROR)   /* Bit 8 of Questionable Status Register */ 
		{
    	bp4462_err = 314;
    	return bp4462_err;
   		}
   if (quest_flag & bp4462_PT100_ERROR)   		/* Bit 9 of Questionable Status Register */ 
		{
    	bp4462_err = 315;
    	return bp4462_err;
   		}
  
  return (0);
}

/*=========================================================================*/
/* This function read the Source Error Id-No.			010 			   */
/*=========================================================================*/
int _VI_FUNC  bp4462_CheckErrorStatusSource (int instr_ID)
{ 
  	int quest_flag;

  	bp4462_err = 0;
   
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if (bp4462_receive_scpi (instr_ID, "S:Q:S?", buf) != 0)
    	return (bp4462_err);
   
	Scan (buf, "%s>%i", &quest_flag);
   
   	if (quest_flag & bp4462_OUTPUT_OVERRANGE_ERROR)   	/* Bit 0 of Questionable Status Output Register */ 
		{
    	bp4462_err = 316;
    	return bp4462_err;
   		}
   	if (quest_flag & bp4462_DA_DEVICE_OVERRANGE_ERROR) 	/* Bit 4 of Questionable Status Output Register */
		{
    	bp4462_err = 318;
    	return bp4462_err;
   		}
   	if (quest_flag & bp4462_PT100_OUTP_ERROR)   		/* Bit 3 of Questionable Status Output Register */    
		{
    	bp4462_err = 317;
    	return bp4462_err;
   		}
   		
  return (0);
}
   		
   		
/*=========================================================================*/
/* This function read the Temperature Error Id-No. 		011     		   */
/*=========================================================================*/
int _VI_FUNC  bp4462_CheckErrorStatusTemp (int instr_ID)
{ 
  	int quest_flag;

  	bp4462_err = 0;

   	/*---- Check the Questionable Status Temperature Register ----*/	
  	if (bp4462_receive_scpi(instr_ID, "S:Q:T?",buf) != 0)
    	return (bp4462_err);
   
   	Scan (buf, "%s>%i",&quest_flag);
  
   	if (quest_flag & bp4462_INVALID_TEMERATURE)  	/* Bit 3 of Questionable Status Temperature Register */    
		{
    	bp4462_err = 323;
    	return bp4462_err;
   		}
   	if (quest_flag & bp4462_PT100_OVERRANGE)		/* Bit 2 of Questionable Status Temperature Register */
		{
    	bp4462_err = 322;
    	return bp4462_err;
   		}
   	if (quest_flag & bp4462_CURR_WIRE_ERROR)		/* Bit 0 of Questionable Status Temperature Register */ 
		{
    	bp4462_err = 320;
    	return bp4462_err;
   		}
   	if (quest_flag & bp4462_VOLT_WIRE_ERROR)   		/* Bit 1 of Questionable Status Temperature Register */    
		{
    	bp4462_err = 321;
    	return bp4462_err;
   		}
   		
  	return (0);
}
   		

/*=========================================================================*/
/* This function read the standard event status register           095 088 */
/*=========================================================================*/
int _VI_FUNC  bp4462_ReadStdEventStatusReg (int instr_ID, int* status, int* bit0, 
											int* bit2, int* bit3, int* bit4, int* bit5)
{
   int quest_flag; 

   bp4462_err = 0;
   
   if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     return bp4462_err;

   if(bp4462_receive_scpi(instr_ID, "*ESR?",buf) != 0)
    return (bp4462_err);

   Scan(buf,"%s>%i[r8]",status);
   quest_flag = *status ;
   
    if(quest_flag & bp4462_OPERATION_COMPLETE) 	/* Bit 0 of Standard Event Status Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
    if(quest_flag & bp4462_QUERY_ERROR)   		/* Bit 2 of Standard Event Status Register */ 
		*bit2 = 1;
	else 
		*bit2 = 0;	
    if(quest_flag & bp4462_DEVICE_DEPENDENT_ERROR)  	/* Bit 3 of Standard Event Status Register */ 
		*bit3 = 1;
	else 
		*bit3 = 0;	
   	if(quest_flag & bp4462_EXECUTION_ERROR)   	/* Bit 4 of Standard Event Status Register */ 
		*bit4 = 1;
	else 
		*bit4 = 0;	
   	if(quest_flag & bp4462_COMMAND_ERROR)   		/* Bit 5 of Standard Event Status Register */ 
		*bit5 = 1;
	else 
		*bit5 = 0;	
		
   return (0);
}
/*=========================================================================*/
/* This function read the questionable status event register     009 096   */
/*=========================================================================*/
int _VI_FUNC  bp4462_ReadQuestStatusEventReg(int instr_ID, int* status, int* bit0, 
											int* bit1, int* bit3, int* bit4, int* bit8, int* bit9)
{ 
  int quest_flag;

  bp4462_err = 0;
   
  if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    return bp4462_err;

  if(bp4462_receive_scpi(instr_ID, "S:Q:E?",buf) != 0)
    return (bp4462_err);
   
   Scan(buf,"%s>%i[r16]",status);
   quest_flag = *status ;
   
   if(quest_flag & bp4462_VOLTAGE_ERROR)   		/* Bit 0 of Questionable Status Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
   if(quest_flag & bp4462_CURRENT_ERROR)   		/* Bit 1 of Questionable Status Register */   
		*bit1 = 1;
	else 
		*bit1 = 0;	
   if(quest_flag & bp4462_POWER_ERROR)   	    /* Bit 3 of Questionable Status Register */ 
		*bit3 = 1;
	else 
		*bit3 = 0;	
   if(quest_flag & bp4462_TEMPERATURE_ERROR)	/* Bit 4 of Questionable Status Register */ 
		*bit4 = 1;
	else 
		*bit4 = 0;	
   if(quest_flag & bp4462_CALIBRATION_ERROR)   	/* Bit 8 of Questionable Status Register */ 
		*bit8 = 1;
	else 
		*bit8 = 0;	
   if(quest_flag & bp4462_PT100_ERROR)   		/* Bit 9 of Questionable Status Register */ 
		*bit9 = 1;
	else 
		*bit9 = 0;	


  return (0);
}

/*=============================================================================*/
/* This function read the questionable status measure temperature register 011 */
/*=============================================================================*/
int _VI_FUNC  bp4462_ReadQuestStatusTempReg (int instr_ID, int* status, int* bit0, 
											int* bit1, int* bit2, int* bit3)
{   
  	int quest_flag;

  	bp4462_err = 0;
   
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if(bp4462_receive_scpi(instr_ID, "S:Q:T?",buf) != 0)
    	return (bp4462_err);
   
   	Scan(buf,"%s>%i", &status);
   	quest_flag = *status ;
   
   	if(quest_flag & bp4462_CURR_WIRE_ERROR)			/* Bit 0 of Questionable Status Temperature Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
   	if(quest_flag & bp4462_VOLT_WIRE_ERROR)   		/* Bit 1 of Questionable Status Temperature Register */    
		*bit1 = 1;
	else 
		*bit1 = 0;	
   	if(quest_flag & bp4462_PT100_OVERRANGE)		   	/* Bit 2 of Questionable Status Temperature Register */
		*bit2 = 1;
	else 
		*bit2 = 0;	
   	if(quest_flag & bp4462_INVALID_TEMERATURE)  	/* Bit 3 of Questionable Status Temperature Register */    
		*bit3 = 1;
	else 
		*bit3 = 0;	

  	return (0);
}

/*================================================================================*/
/* This function read the questionable status measure Output register     010     */
/*================================================================================*/
int _VI_FUNC  bp4462_ReadQuestStatusOutputReg (int instr_ID, int* status, int* bit0, 
												int* bit3, int* bit4)
{	  
  	int quest_flag;

  	bp4462_err = 0;
   
  	if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    	return bp4462_err;

  	if(bp4462_receive_scpi(instr_ID, "S:Q:S?",buf) != 0)
    	return (bp4462_err);
   
   	Scan(buf,"%s>%i", &status);
   	quest_flag = *status ;
   
   	if(quest_flag & bp4462_OUTPUT_OVERRANGE_ERROR)   /* Bit 0 of Questionable Status Output Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
   	if(quest_flag & bp4462_PT100_OUTP_ERROR)   		 /* Bit 3 of Questionable Status Output Register */    
		*bit3 = 1;
	else 
		*bit3 = 0;	
   	if(quest_flag & bp4462_DA_DEVICE_OVERRANGE_ERROR) /* Bit 4 of Questionable Status Output Register */
		*bit4 = 1;
	else 
		*bit4 = 0;	
  
  	return (0);
}

/*=========================================================================*/
/* This function read the operation status condition register     007      */
/*=========================================================================*/
int _VI_FUNC  bp4462_ReadOperStatusCondReg (int instr_ID, int* status, int* bit0, int* bit8)
{
  int quest_flag;

  bp4462_err = 0;
   
  if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
    return bp4462_err;

  if(bp4462_receive_scpi(instr_ID, "S:O:C?",buf) != 0)
    return (bp4462_err);

   	Scan(buf,"%s>%i", &quest_flag);
   	*status = quest_flag;
   
   	if(quest_flag & bp4462_CALEBRATING_STATE) 		/* Bit 0 of Operation Status Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
   	if(quest_flag & bp4462_CAL_MEASURE_FLAG) 		/* Bit 8 of Operation Status Register */    
		*bit8 = 1;
	else
		*bit8 = 0;	

  	return (0);
}

/*=========================================================================*/
/* This function read the operation status event register         008      */
/*=========================================================================*/
int _VI_FUNC  bp4462_ReadOperStatusEventReg (int instr_ID, int* status, int* bit0, int* bit8)
{
   int quest_flag;

   bp4462_err = 0;
   
   if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     return bp4462_err;

   if(bp4462_receive_scpi(instr_ID, "S:O:E?",buf) != 0)
    return (bp4462_err);

   	Scan(buf,"%s>%i[r16]",status);
   	quest_flag = *status ;
   
   	if(quest_flag & bp4462_CALEBRATING_STATE) 		/* Bit 0 of Operation Status Register */ 
		*bit0 = 1;
	else 
		*bit0 = 0;	
   	if(quest_flag & bp4462_CAL_MEASURE_FLAG) 		/* Bit 8 of Operation Status Register */    
		*bit8 = 1;
	else 
		*bit8 = 0;	

   return (0);
}


/*===============================================================================*/
/* This function reset the operation and questionable status enable register 012 */
/*===============================================================================*/
int _VI_FUNC  bp4462_ResetQuestOperStEnReg(int instr_ID)
{
  bp4462_err = 0;
   
  if(bp4462_send_scpi(instr_ID, "STAT:PRES") != 0)
    return (bp4462_err);

  return (0);
}



/***************************** Interface ******************************************/

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC  bp4462_send_rs232(int instr_ID, char* msg)
{
    int ret,r_byte,l;
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     return bp4462_err;
    
    FlushInQ(port_no);           			/* Clear Port */
    
    
    Fmt(out_buf,"%c%s%c",STX,msg,ETX);     	/* String set*/
    
    l = StringLength(out_buf);
    ret = ComWrt(port_no, out_buf, l);     	/* Write the Message */
    if (rs232err != 0)
     return (230);
    r_byte = ComRdByte(port_no);			/* read the answer of slave*/
    if (rs232err != 0)
     return (231);				

	/*FmtOut("\nsend %s ret %d \n",out_buf,r_byte);*/
    
    switch(r_byte)					/*Witch Answer ?*/
    {
      case ACK:
       return  (0);					/* ACK => all ready*/
       
      case NAK:
       return (300);                /* NAK Error-Code */
       
      case EOT:						
       return (301);				/* EOT Error Code*/
       
    }

    return (0);
}

/*=========================================================================*/
/* This function receive a string            							   */
/*=========================================================================*/
int _VI_FUNC  bp4462_receive_rs232(int instr_ID, char *cmd, char* receive_buf)
{
    int ret;
    
    
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     return bp4462_err;

	/*FmtOut("\n++send '%s'",cmd);*/
    bp4462_err = bp4462_send_rs232(instr_ID, cmd);
    if(bp4462_err != 0)
     return (bp4462_err);
     
    FlushInQ(port_no);           /* Clear Port */
    
    ret = bp4462_receive_rs232_message(instr_ID, receive_buf);
    if(ret != 0)
     return (ret);
    
     
    return(0);              				 /* 0 = ACK == OK*/
	
}
/*=========================================================================*/
/* This function receive a string          4462  							   */
/*=========================================================================*/
int _VI_FUNC  bp4462_receive_rs232_message(int instr_ID, char* receive_buf)
{
    int first_byte , r_byte,l;
    
    
    
    if (bp4462_invalid_integer_range (instr_ID, 1, bp4462_MAX_INSTR, -1) != 0)
     return bp4462_err;

	ComWrtByte(port_no,EOT);/* send EOT to slave => Master is ready to read */
	
    first_byte = ComRdByte(port_no);   /* Read the EOT or STX from Port*/
    /*FmtOut("\n**First '%d'",first_byte);*/
    if(first_byte == EOT)
    {
   	/*FmtOut("\n**First '%d'",first_byte);*/
     	return (301);
    } 

    r_byte = ComRdTerm(port_no,in_buf,bp4462_MAX_CMD-1,ETX);/* Read to ETX*/
    if (rs232err != 0)
     return (231);
 
    in_buf[r_byte] = 0x00;				 /* terminate the string*/
    
    l = StringLength(in_buf);     
    
    /*FmtOut("\n++ inbuf '%s'",in_buf);*/
    
 /****/   CopyString (receive_buf, 0,in_buf,0,l-2); /* Copy without LF,ETX*/
    ComWrtByte(port_no,ACK);	 /* Send ACK to slave => all ready */
    
    r_byte = ComRdByte(port_no);   /* Read the EOT from Port => End of Transmission*/
    /*FmtOut("\n**last '%c'",r_byte);*/
    if(r_byte == EOT)
     return (0);							 /* No more Datas to read*/
    /*else
     return (309);							  /* More Datas  to read*/
    
    return(0);   
}

/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC  bp4462_close (int instr_ID)
{
    bp4462_err = 0;

    if(interface == bp4462_GPIB)  /* GPIB ?? */
    {
     if (bp4462_device_closed_gpib (instr_ID) != 0)
      return (bp4462_err);
     bp4462_close_instr_gpib (instr_ID);
    }
    else
    {
     bp4462_close_instr_serial (instr_ID);
    }
    return (bp4462_err);
}
/* = GPIB UTILITY ROUTINES == ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the GPIB device table for the instrument.  If successful, the        */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2329_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC bp4462_open_instr_gpib(int addr)
{
    int i;
    int instr_ID = 0;
    
    /* Check to see if the instrument is already in the Instrument Table. */
    for (i=1; i<= bp4462_MAX_INSTR; i++)
    if (address[i] == addr)
    {
      instr_ID = i;
      i=bp4462_MAX_INSTR;
    }
    /* If it is not in the instrument Table, open an entry for the instrument */
    if (instr_ID <= 0)
    for(i=1; i<=bp4462_MAX_INSTR; i++)
      if(!address[i])
      {
      instr_ID = i;
      i= bp4462_MAX_INSTR;
      }
	/* If an entry could not be opened in the Instrument Table, return an error. */
	if (instr_ID<= 0)
	{
	bp4462_err = 220;
	return -1;
	}
    /*   If the device has not been opened in the GPIB device table (bd = 0), */
    /*   then open it.                                                        */
    if (bd[instr_ID] <= 0)
    {
    if (instr_cnt<=0)
      CloseInstrDevs("bp4462");
    bd[instr_ID] = OpenDev ("", "bp4462");
    if (bd[instr_ID] <= 0)
    {
        bp4462_err = 220;
        return  -1;
    }
    }
    /*   Change the primary address of the device     */
    if (ibpad (bd[instr_ID], addr) < 0)
    {
    bp4462_err = 233;
    return  -1;
    }
    address[instr_ID]=addr;
    instr_cnt++;
    
    return instr_ID;
}

/* ========================================================================== */
/*  This function closes the instrument by removing it from the GPIB device   */
/*  table and setting the address and the bd to zero in the Instrument Table. */
/*  The return value is equal to the global error variable.                   */
/* ========================================================================== */
int _VI_FUNC  bp4462_close_instr_gpib (int instr_ID)
{
    if (bd[instr_ID] != 0) 
    	{
    	CloseDev (bd[instr_ID]);
    	bd[instr_ID] = 0;
    	address[instr_ID] = 0;
    	}
    else
    	bp4462_err = 221;
    return bp4462_err;
}
/* ========================================================================= */
/*  This function checks to see if the module has been initialized.  If the  */
/*  device has not been opened, a -1 is returned, 0 otherwise.               */
/* ========================================================================= */
int _VI_FUNC  bp4462_device_closed_gpib (int instr_ID)
{
    if (bd[instr_ID] <= 0)
    {
    bp4462_err = 232;
    return  -1;
    }
    return 0;
}

/* ========================================================================= */
/*  This function reads a buffer of data from the instrument. The return     */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp4462_read_msg_gpib (int instr_ID, char *msg,int cnt)
{
    ibrd (bd[instr_ID], msg, (long)cnt);
    

    if(iberr != 0)
        bp4462_err = 231;
        
    else
        bp4462_err = 0;

	if(ibcntl >= 2)
     msg[ibcntl-2] = 0x00;
     
    /*FmtOut("\n%s",msg);*/

    return bp4462_err;
}

/* ========================================================================= */
/*  This function writes a buffer of data to the instrument. The return      */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp4462_write_msg_gpib (int instr_ID, char *msg,int cnt)
{
	
/*    FmtOut("\n%s",msg);*/
	
	(ibwrt (bd[instr_ID], msg, (long)cnt));
	
     if (iberr != 0)				  
     bp4462_err = 230;
    else
     bp4462_err = 0;
     
    return bp4462_err;
}
/* = SERIAL UTILITY ROUTINES ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the            */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2329_MAX_INSTR.                                               */
/* ========================================================================= */
/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function opens and configures the selected COM port      */
/*           The return value is equal to the global error variable.       */
/*=========================================================================*/
int _VI_FUNC bp4462_open_instr_serial (int port, long baud_r, int parity, int datab, int stopb)
{
	bp4462_err = 0;
  
	bp4462_err = OpenComConfig (port, "", baud_r, parity, datab, stopb, 512, 512);
     	if (bp4462_err != 0)
           	return -1;
        else return (1);
}

/*=========================================================================*/
/* Function: Close Instrument Serial                                       */
/* Purpose:  This function closes the port for the instrument module and   */
/*           sets the port to zero. The return value is equal to the       */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC bp4462_close_instr_serial (int instr_ID)
{
/*  Close the com port.  If error, set bp2329_err = rs232err  */
    CloseCom (port_no);
    if (rs232err != 0)
     	bp4462_err = rs232err;

    return bp4462_err;
}

/*=========================================================================*/
/* Function: Invalid Real Range            4462                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp4462_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp4462_err = err_code;
    return -1;
  }
  return 0;
}

/* = Global UTILITY ROUTINES ==============================================*/
/*=========================================================================*/
/* Function: Invalid Integer Range          4462                               */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp4462_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp4462_err = err_code;
    return -1;
  }
  return 0;
}



/***************************************************************************/

void bp4462_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
    for (i=1; i>=bp4462_MAX_INSTR; i++)
     {
      dev_port[i] = 0;
      address[i] = 0;
     }
     setup_first = 1;
     instr_cnt = -1;
    }
    
    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

    step_mode[0] = "ramp:step down";
    step_mode[1] = "ramp:step up";
    
    ramp_start[0] = "ramp:stop";
    ramp_start[1] = "ramp:start";
    
    sense_mode[0] = "sour:volt:alc:sour int";
    sense_mode[1] = "sour:volt:alc:sour ext";
    
    display_mode[0] = "DISP:MENU DX";
    display_mode[1] = "DISP:MENU RJTEMP";
    display_mode[2] = "DISP:MENU UT";
    display_mode[3] = "DISP:MENU U0";
    display_mode[4] = "DISP:MENU SOUR";
    display_mode[5] = "DISP:MENU PT100";
    
    output_mode[0] = "SOUR:MODE FIX";
    output_mode[1] = "SOUR:MODE SWE";
    output_mode[2] = "SOUR:MODE LIST";
    
    tc[0]    = "CONF:TEMP:TCO R";
    tc[1]    = "CONF:TEMP:TCO S";
    tc[2]    = "CONF:TEMP:TCO B";
    tc[3]    = "CONF:TEMP:TCO J";
    tc[4]    = "CONF:TEMP:TCO T";
    tc[5]    = "CONF:TEMP:TCO E";
    tc[6]    = "CONF:TEMP:TCO K";
    tc[7]    = "CONF:TEMP:TCO U";
    tc[8]    = "CONF:TEMP:TCO L";
    tc[9]    = "CONF:TEMP:TCO N";
    tc[10]    = "CONF:TEMP:TCO C";
    tc[11]    = "CONF:TEMP:TCO D";
    tc[12]    = "CONF:TEMP:TCO G2";
    tc[13]    = "CONF:TEMP:TCO M";

    tc_unit[0]  = "UNIT:TEMP:TCO C";
    tc_unit[1]  = "UNIT:TEMP:TCO F";
    tc_unit[2]  = "UNIT:TEMP:TCO K";
    
    tc_scale[0] = "CONF:TEMP:SCAL IPTS68";
    tc_scale[1] = "CONF:TEMP:SCAL ITS90";
    
    ramp1_wave[0] = "SOUR:SWE:WAV SAWT";
    ramp1_wave[1] = "SOUR:SWE:WAV TRI";
    
    ramp2_wave[0] = "sour:list:wav sawt";
    ramp2_wave[1] = "sour:list:wav tri";
    
    unit_type[0] = "UV";
    unit_type[1] = "MV";
    unit_type[2] = "V";
    unit_type[3] = "UA";
    unit_type[4] = "MA";
    unit_type[5] = "C";
    unit_type[6] = "F";
    unit_type[7] = "K";
    
    
    state_str[0] = "LOW";
    state_str[1] = "HIGH";  
    
    unit_str[0] = "VOLT";
    unit_str[1] = "CURR";
  
	range_str[0] = "VOLT 300MV";
	range_str[1] = "VOLT 3V";
	range_str[2] = "VOLT 30V";
	range_str[3] = "VOLT 60V";
	
	
	sign_str[0] = "NEG";
	sign_str[1] = "POS";
	
	output_str[0] = "ZERO";
	output_str[1] = "LOW";  
	output_str[2] = "HIGH";  
	
    rj_type[0] = "SENS:TCO:REFJ RJ-MAN";
    rj_type[1] = "SENS:TCO:REFJ RJ-EXT";
    
    man_range[0] = "SOUR:VOLT:RANG 300MV";
    man_range[1] = "SOUR:VOLT:RANG 3V";
    man_range[2] = "SOUR:VOLT:RANG 30V";   
    man_range[3] = "SOUR:VOLT:RANG 60V";   
    
    set_delta[0] = "Sour:DELT:SUB";
    set_delta[1] = "Sour:DELT:ADD";
    
    delta_value[0] = "SOUR:VOLT:DELT?";
    delta_value[1] = "SOUR:CURR:DELT?";
    delta_value[2] = "SOUR:TCO:DELT?";
    
}


/*--------- String converting functions ----------------*/

/* ====================================================================================== */
/*  This function demounts the originstring into 3 valuestrings (numeral characters 1-9)  */
/*  and in 3 unitstrings (textual characters A-Z); the characters ',', '+' and ' '        */
/*  will be ignored.	        														  */
/* ====================================================================================== */

void _VI_FUNC  bp4462_strsplit (char originstr[], char *valuestr1, char *unitstr1, char *valuestr2, char *unitstr2, char *valuestr3, char *unitstr3)
{				   
  int i = 0, z;

  /* While ',' , ' ' or '+' */
  while ((originstr[i] == 44)||(originstr[i] == 32)||(originstr[i] == 43)) ++i;
  /* While numbers, '.' or '-' */ 
  for(z=0; ((originstr[i] >= 48) && (originstr[i] <= 57))||(originstr[i] == 46)||(originstr[i] == 45); z++, i++)
  valuestr1[z] = originstr[i];
  if (originstr[i] == 0)
  	return;  
  valuestr1[z] = 0;
  /* While ',' or ' ' */
  if ((originstr[i] == 44)||(originstr[i] == 32)) 
    ++i;
  /* While letters */  
  for(z=0; (originstr[i] >= 65) && (originstr[i] <= 90) ; z++,i++)
  unitstr1[z] = (originstr[i]);
  if (originstr[i] == 0)
   	return;  
  unitstr1[z] = 0;
  /* While ',' , ' ' or '+' */ 
  if ((originstr[i] == 44)||(originstr[i] == 32)||(originstr[i] == 43)) 
    ++i;
  /* While numbers, '.' or '-' */
  for(z=0; (originstr[i] >= 48) && (originstr[i] <= 57)||(originstr[i] == 46)||(originstr[i] == 45); z++,i++)
  valuestr2[z] = originstr[i];
  if (originstr[i] == 0)
   	return;  
  valuestr2[z] = 0;
  /* While ',' , ' ' or '+' */
  if ((originstr[i] == 44)||(originstr[i] == 32)||(originstr[i] == 43)) 
    ++i;
  /* While letters */  
  for(z=0; (originstr[i] >= 65) && (originstr[i] <= 90) ; z++,i++)
  unitstr2[z] = originstr[i];
  if (originstr[i] == 0)
   	return;  
  unitstr2[z] = 0;
  /* While ',' , ' ' or '+' */
  if ((originstr[i] == 44)||(originstr[i] == 32)||(originstr[i] == 43)) 
    ++i;
  /* While numbers, '.' or '-' */
  for(z=0; (originstr[i] >= 48) && (originstr[i] <= 57)||(originstr[i] == 46)||(originstr[i] == 45); z++,i++)
  valuestr3[z] = originstr[i];
  if (originstr[i] == 0)
   	return;  
  valuestr3[z] = 0;
  /* While ',' , ' ' or '+' */
  if ((originstr[i] == 44)||(originstr[i] == 32)||(originstr[i] == 43)) 
    ++i;
  /* While letters */
  for(z=0; (originstr[i] >= 65) && (originstr[i] <= 90) ; z++,i++)
  unitstr3[z] = originstr[i];
  if (originstr[i] == 0)
    return; 
  unitstr3[z] = 0;
  
return;
}

/* ==================================================================================== */
/*  This function splits five double-values from a string                               */
/* ==================================================================================== */

void _VI_FUNC bp4462_valuesplit(char originstr[], double *value1, double *value2, 
								double *value3, double *value4, double *value5)
{				   
  	int i = 0, z = 0, s = 0, w = 1, c, flag;
  	char dummy[80];
  	char valuestr[10][80];
  	
    while (w == 1)
  		{
  		/* If ',', ' ' or 0 */
  		if ((originstr[z] == 44)||(originstr[z] == 32)||(originstr[z] == 0)) 
  			{
  			if (flag == 1)
  				{
  				++i;
  				++s;
  				dummy[i] = 0;
  				for (c=0; c<i; ++c)
  					{
  					valuestr[s][c] = dummy[c];
  					}
  				i = 0;
  				flag = 0;
  				}
  			}
  		else
  			{
  			flag = 1;
  			dummy[i] = originstr[z];
  			++i;
  			}
  		if (s>3)
  			if (originstr[z] == 0)
  				w=0;
  		++z;
  		}
  	/* Format the values */	
  	Fmt(value1,"%f<%s",valuestr[1]);	
	Fmt(value2,"%f<%s",valuestr[2]);
	Fmt(value3,"%f<%s",valuestr[3]);
	Fmt(value4,"%f<%s",valuestr[4]);
	Fmt(value5,"%f<%s",valuestr[5]);
	
	return;
}	
	

/*= THE END ===============================================================*/

